c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
c $Rev: 54 $ $Date: 2014-12-31 12:05:49 -0500 (Wed, 31 Dec 2014) $
c FORTRAN 77
c ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      PROGRAM  TESTDO

c     Runs unit test problems for DISORT and checks answers. These
c     problems test almost all logical branches in DISORT.
c
c     By default, run all unit tests. To run a subset of unit tests,
c     change the DATA statement for DOPROB; e.g. to run just problem 4,
c     set DOPROB(4) .TRUE. and all others .FALSE.
c
c     Version 3 upgrades:
c          
c     Added new test problems:   
c          
c     1) Problem 14: BRDF tests: check that reflected intensities match
c                    analytic BRDF results.
c                   (DISORT setup: Beam incidence, layer=vacuum, BRDF)
c
c     2) Problem 15: a demo showing how to setup DISORT3 for
c                    multiple layers + BRDF system.
c                    (DISORT setup: Beam incidence, 1 Rayleigh layer,
c                    1 aerosol layer, BRDF)
c
c     It is HIGHLY recommended that you use the code below as a template
c     for creating your own CALLs to DISORT, rather than starting from
c     scratch.  This will prevent a lot of mistakes and ensure that every
c     input argument gets a value.  Note in particular how GETMOM is
c     sometimes called to fill an array section of PMOM (for one layer);
c     several people have done this incorrectly in attempting to write it
c     ab initio (passing array sections for arrays that do not start at
c     element 1 is tricky).
c
c     Note that the ratio to the 'correct answer' may occasionally be
c     significantly different from unity -- even so different that
c     the ratio just prints as ****** rather than a number.  However,
c     this mostly occurs for values of flux or intensity that are very
c     small compared to the forcing functions (that is, small compared
c     to internal thermal emission and/or radiation incident at the
c     boundaries).  The printed number 'SERIOUSLY NON-UNIT RATIOS'
c     attempts to count just the cases where there is a real disagreement
c     and not those where quantitites are down at their noise level
c     (defined as 10^(-6) times their maximum value).
c
c     Further documentation can be found in the file DISOTEST.txt.
c
c  Routines called :
c
c    DISORT:   The discrete ordinates radiative transfer program
c
c    BDREF:    Sets bidirectional reflectance of lower boundary
c
c    GETMOM:   Sets phase function Legendre coefficients
c
c    PRTFIN:   Prints fluxes and intensities and their ratios to
c              the correct values
c
c    CHEKDO:   Data block containing correct fluxes and intensities
c
c    RATIO :   Ratio of calculated to correct value with underflow
c              and overflow protection (kept in file DISORT.f)
c
c+---------------------------------------------------------------------+
c     ** DISORT I/O specifications
      INTEGER  MAXCLY, MAXMOM, MAXPHI, MAXULV, MAXUMU
      PARAMETER ( MAXCLY = 6, MAXMOM = 999, MAXPHI = 3, MAXULV = 5,
     &            MAXUMU = 10 )
      CHARACTER  HEADER*127
      LOGICAL  LAMBER, PLANK, ONLYFL, PRNT(5), USRANG, USRTAU
      LOGICAL  DEBUG
      INTEGER  IBCND, NMOM, NLYR, NUMU, NSTR, NPHI, NTAU
      REAL     ACCUR, ALBEDO, BTEMP, DTAUC( MAXCLY ), FBEAM, FISOT,
     &         PHI( MAXPHI ), PMOM( 0:MAXMOM, MAXCLY ),
     &         PHI0, SSALB( MAXCLY ), TEMPER( 0:MAXCLY ), TEMIS, TTEMP,
     &         WVNMLO, WVNMHI, UMU( MAXUMU ), UMU0, UTAU( MAXULV )

      REAL     RFLDIR( MAXULV ), RFLDN( MAXULV ), FLUP( MAXULV ),
     &         DFDT( MAXULV ), UAVG( MAXULV ),
     &         UU( MAXUMU, MAXULV, MAXPHI ), ALBMED( MAXUMU ),
     &         TRNMED( MAXUMU )

c+---------------------------------------------------------------------+
c     ** Correct answers
      INTEGER  MXPROB, MXCASE, MXTAU, MXMU, MXPHI
      PARAMETER  ( MXPROB = 15, MXCASE = 8, MXTAU = 5, MXMU = 15,
     &             MXPHI = 3 )
      REAL  TSTFIR( MXTAU, MXCASE, MXPROB ),
     &      TSTFDN( MXTAU, MXCASE, MXPROB ),
     &      TSTFUP( MXTAU, MXCASE, MXPROB ),
     &      TSTDFD( MXTAU, MXCASE, MXPROB ),
     &      TSTUU ( MXTAU, MXMU, MXPHI, MXCASE, MXPROB )
      COMMON / DOCHEK / TSTFIR, TSTFDN, TSTFUP, TSTDFD, TSTUU

c+---------------------------------------------------------------------+
c     ** DISOBRDF input
c     ** Parameter constants must match ones inside DISORT

      CHARACTER*1  ABC(18)*1, TITLE*100, BLANKS*3
      LOGICAL      DOPROB( 15 )
      INTEGER      ICAS, IOD, ISS, IU, J, K, LC, LENTIT, LU, NPROB
      REAL         CMPFIR( MXTAU ), CMPFDN( MXTAU ), CMPFUP( MXTAU ),
     &             CMPDFD( MXTAU ), CMPUU ( MXTAU, MXMU, MXPHI ),
     &              PI
      PARAMETER     ( PI = 2.*ASIN(1.0) )

      INTEGER       MXCMU, MI, NAZZ
      PARAMETER     (MXCMU = 48, MI = MXCMU/2, 
     &              NAZZ = MXCMU -1 ) 
      REAL      RHOQ(MI, 0:MI, 0:NAZZ), RHOU(MAXUMU, 0:MI, 0:NAZZ),
     &          EMUST(MAXUMU), BEMST(MI)
      REAL      BDR_BEAM_ANALYTIC(MAXUMU, MAXPHI)

      INTEGER       NPASS
      INTEGER       BRDF_TYPE
      REAL          BRDF_ARG(4)
      LOGICAL       DO_SHADOW
      REAL          WIND_SPD, REFRAC_INDEX
      REAL          B0, HH, W
      REAL          K_VOL, K_ISO, K_GEO
      REAL          RHO_0, KAPPA, G, H0
      REAL          FLUX_UP, DFDTAU
      INTEGER       NMUG
c     ..
c     .. External Subroutines ..
      EXTERNAL  DISORT, ERRMSG, GETMOM, PRTFIN, PRTFIN2
c     ..
c     .. Intrinsic Functions ..
      INTRINSIC ASIN, FLOAT, INDEX
c     ..
      DATA  ABC / 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j',
     &            'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r' /,
     &      PRNT / .TRUE., 3*.FALSE., .TRUE. /, ACCUR / 0.0 /,
     &      BLANKS / '   ' /,  DOPROB / 15*.TRUE. /

      NTEST = 0
      NPASS = 0

c      DOPROB(1) = .TRUE.
c      DOPROB(1) = .FALSE.
      
      IF( DOPROB(1) )  THEN

c **********************************************************************
c ****  Test Problem 1:  Isotropic Scattering                       ****
c ****  (Compare to Ref. VH1, Table 12)                             ****
c **********************************************************************

      NSTR = 16
      NLYR = 1
      NMOM = NSTR
      CALL  GETMOM( 1, 0.0, NMOM, PMOM )
      USRTAU    = .TRUE.
      NTAU      = 2
      UTAU( 1 ) = 0.0
      USRANG    = .TRUE.
      NUMU      = 6
      UMU( 1 )  = -1.0
      UMU( 2 )  = -0.5
      UMU( 3 )  = -0.1
      UMU( 4 )  =  0.1
      UMU( 5 )  =  0.5
      UMU( 6 )  =  1.0
      NPHI      = 1
      PHI( 1 )  = 0.0
      IBCND     = 0
      UMU0      = 0.1
      PHI0      = 0.0
      LAMBER    = .TRUE.
      ALBEDO    = 0.0
      PLANK     = .FALSE.
      ONLYFL    = .FALSE.

      DO 10  ICAS = 1,6 

         IF ( ICAS.EQ.1 ) THEN

            UTAU( 2 )  = 0.03125
            SSALB( 1 ) = 0.2
            FBEAM      = PI / UMU0
            FISOT      = 0.0

         ELSE IF ( ICAS.EQ.2 ) THEN

            UTAU( 2 )  = 0.03125
            SSALB( 1 ) = 1.0
            FBEAM      = PI / UMU0
            FISOT      = 0.0

         ELSE IF ( ICAS.EQ.3 ) THEN

            UTAU( 2 )  = 0.03125
            SSALB( 1 ) = 0.99
            FBEAM      = 0.0
            FISOT      = 1.0

         ELSE IF ( ICAS.EQ.4 ) THEN

            UTAU( 2 )  = 32.0
            SSALB( 1 ) = 0.2
            FBEAM      = PI / UMU0
            FISOT      = 0.0

         ELSE IF ( ICAS.EQ.5 ) THEN

            UTAU( 2 )  = 32.0
            SSALB( 1 ) = 1.0
            FBEAM      = PI / UMU0
            FISOT      = 0.0

         ELSE IF ( ICAS.EQ.6 ) THEN

            UTAU( 2 )  = 32.0
            SSALB( 1 ) = 0.99
            FBEAM      = 0.0
            FISOT      = 1.0

         END IF

         DTAUC( 1 ) = UTAU( 2 )

         WRITE( HEADER,'(3A,F9.5,A,F5.2)') 'Test Case No. 1',ABC(ICAS),
     &          ':  Isotropic Scattering, Ref. VH1, Table 12:  b =',
     &          UTAU(2), ', a =', SSALB(1)

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 1
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &       ERRMSG( 'Out of bounds in exact-answer arrays', .FALSE. )

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
  10  CONTINUE

      ENDIF

c      DOPROB(2) = .TRUE.
c      DOPROB(2) = .FALSE.

      IF( DOPROB(2) )  THEN

c **********************************************************************
c ****  Test Problem 2:  Rayleigh Scattering, Beam Source           ****
c ****  (Compare To Ref. SW, Table 1)                               ****
c **********************************************************************

      NSTR = 16
      NLYR = 1
      NMOM = NSTR
      CALL  GETMOM( 2, 0.0, NMOM, PMOM )
      USRTAU    = .TRUE.
      NTAU      = 2
      UTAU( 1 ) = 0.0
      USRANG    = .TRUE.
      NUMU      = 6
      UMU( 1 )  = -0.981986
      UMU( 2 )  = -0.538263
      UMU( 3 )  = -0.018014
      UMU( 4 )  =  0.018014
      UMU( 5 )  =  0.538263
      UMU( 6 )  =  0.981986
      NPHI      = 1
      PHI( 1 )  = 0.0
      IBCND     = 0
      FBEAM     = PI
      UMU0      = 0.080442
      PHI0      = 0.0
      FISOT     = 0.0
      LAMBER    = .TRUE.
      ALBEDO    = 0.0
      PLANK     = .FALSE.
      ONLYFL    = .FALSE.

      ICAS = 0
      DO 20  IOD = 1, 2

         IF ( IOD.EQ.1 )  UTAU( 2 ) = 0.2
         IF ( IOD.EQ.2 )  UTAU( 2 ) = 5.0

         DTAUC( 1 ) = UTAU( 2 )

         DO 20  ISS = 1, 2

            IF( ISS.EQ.1 )  SSALB( 1 ) = 0.5
            IF( ISS.EQ.2 )  SSALB( 1 ) = 1.0
            ICAS = ICAS + 1

            WRITE( HEADER, '(3A,F5.2,A,F9.6,A,F4.2)')
     &             'Test Case No. 2', ABC(ICAS),
     &             ', Rayleigh Scattering, Ref. SW, Table 1:  tau =',
     &             UTAU(2), ', mu0 =', UMU0, ', ss-albedo =', SSALB(1)

            CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                    WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                    USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                    UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                    TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                    HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                    MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                    ALBMED, TRNMED, 
     &                    BEMST, EMUST, RHOQ, RHOU,
     &                    BDR_BEAM_ANALYTIC )    

            NPROB = 2
            IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

            CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                    MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                    TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                    TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                    TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU,
     &                    MXPHI, NTEST, NPASS )
   20 CONTINUE

      ENDIF

c      DOPROB(3) = .TRUE.
c      DOPROB(3) = .FALSE.

      IF( DOPROB(3) )  THEN

c **********************************************************************
c ****  Test Problem 3:  Henyey-Greenstein Scattering               ****
c ****  (Compare To Ref. VH2, Table 37)                             ****
c **********************************************************************

      NSTR = 16
      NLYR = 1
      SSALB( 1 ) = 1.0
      NMOM = 32
      CALL  GETMOM( 3, 0.75, NMOM, PMOM )
      USRTAU    = .TRUE.
      NTAU      = 2
      UTAU( 1 ) = 0.0
      USRANG    = .TRUE.
      NUMU      = 6
      UMU( 1 )  = -1.0
      UMU( 2 )  = -0.5
      UMU( 3 )  = -0.1
      UMU( 4 )  =  0.1
      UMU( 5 )  =  0.5
      UMU( 6 )  =  1.0
      NPHI      = 1
      PHI( 1 )  = 0.0
      IBCND     = 0
      UMU0      = 1.0
      PHI0      = 0.0
      FBEAM      = PI / UMU0
      FISOT      = 0.0
      LAMBER    = .TRUE.
      ONLYFL    = .FALSE.
      ALBEDO    = 0.0
      PLANK     = .FALSE.

      DO 30  ICAS = 1, 2

         IF ( ICAS.EQ.1 ) THEN

            UTAU( 2 )  = 1.0

         ELSE IF ( ICAS.EQ.2 ) THEN

            UTAU( 2 )  = 8.0

         END IF

         DTAUC( 1 ) = UTAU( 2 )

         WRITE( HEADER, '(3A,F9.5,A,F5.2)') 'Test Case No. 3',ABC(ICAS),
     &         ', Henyey-Greenstein Scattering, Ref. VH2, Table 37,'
     &         //' g = 0.75, b =', UTAU(2), ', a =', SSALB(1)

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 3
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
30    CONTINUE

      ENDIF

c      DOPROB(4) = .TRUE.
c      DOPROB(4) = .FALSE.

      IF( DOPROB(4) )  THEN

c **********************************************************************
c ****  Test Problem 4:  Haze-L Scattering, Beam Source             ****
c ****  (Compare to Ref. GS, Tables 12-16)                          ****
c **********************************************************************

      NSTR = 32
      NLYR = 1
      NMOM = NSTR
      CALL  GETMOM( 4, 0.0, NMOM, PMOM )
      DTAUC( 1 ) = 1.0
      USRTAU     = .TRUE.
      NTAU       = 3
      UTAU( 1 )  = 0.0
      UTAU( 2 )  = 0.5
      UTAU( 3 )  = 1.0
      USRANG     = .TRUE.
      NUMU       = 6
      UMU( 1 )   = -1.0
      UMU( 2 )   = -0.5
      UMU( 3 )   = -0.1
      UMU( 4 )   =  0.1
      UMU( 5 )   =  0.5
      UMU( 6 )   =  1.0
      IBCND      = 0
      FBEAM      = PI
      PHI0       = 0.0
      FISOT      = 0.0
      LAMBER     = .TRUE.
      ALBEDO     = 0.0
      PLANK      = .FALSE.
      ONLYFL     = .FALSE.

      DO 40 ICAS = 1, 3

         WRITE( TITLE, '(3A)' ) 'Test Case No. 4', ABC(ICAS),
     &          ', Haze-L Scattering, Ref. GS, Table '
         LENTIT = INDEX( TITLE,BLANKS )

         IF ( ICAS.EQ.1 ) THEN

            SSALB( 1 ) = 1.0
            NPHI       = 1
            PHI( 1 )   = 0.0
            UMU0       = 1.0
            HEADER = TITLE(1:LENTIT) // ' 12'

         ELSE IF ( ICAS.EQ.2 ) THEN

            SSALB( 1 ) = 0.9
            NPHI       = 1
            PHI( 1 )   = 0.0
            UMU0       = 1.0
            HEADER = TITLE(1:LENTIT) // ' 13'

         ELSE IF ( ICAS.EQ.3 ) THEN

            SSALB( 1 ) = 0.9
            NPHI       = 3
            PHI( 1 )   = 0.0
            PHI( 2 )   = 90.0
            PHI( 3 )   = 180.0
            UMU0       = 0.5
            HEADER = TITLE(1:LENTIT) // ' 14-16'

         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 4
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
40    CONTINUE

      ENDIF

c      DOPROB(5) = .TRUE.
c      DOPROB(5) = .FALSE.

      IF( DOPROB(5) )  THEN

c **********************************************************************
c ****  Test Problem 5:  Cloud C.1 Scattering, Beam Source          ****
c ****  (Compare to Ref. GS, Tables 19-20)                          ****
c **********************************************************************

      NSTR = 48
      NLYR = 1
      NMOM = 299
      CALL  GETMOM( 5, 0.0, NMOM, PMOM )
      DTAUC( 1 ) = 64.0
      USRTAU     = .TRUE.
      NTAU       = 3
      USRANG     = .TRUE.
      NUMU       = 6
      UMU( 1 )   = -1.0
      UMU( 2 )   = -0.5
      UMU( 3 )   = -0.1
      UMU( 4 )   =  0.1
      UMU( 5 )   =  0.5
      UMU( 6 )   =  1.0
      NPHI       = 1
      PHI( 1 )   = 0.0
      IBCND      = 0
      FBEAM      = PI
      UMU0       = 1.0
      PHI0       = 0.0
      FISOT      = 0.0
      LAMBER     = .TRUE.
      ALBEDO     = 0.0
      PLANK      = .FALSE.
      ONLYFL     = .FALSE.

      DO 50 ICAS = 1, 2

         WRITE( TITLE, '(3A)' ) 'Test Case No. 5', ABC(ICAS),
     &          ', Cloud C.1 Scattering, Ref. GS, Table '
         LENTIT = INDEX( TITLE,BLANKS )

         IF ( ICAS.EQ.1 ) THEN

            UTAU( 1 )  = 0.0
            UTAU( 2 )  = 32.0
            UTAU( 3 )  = 64.0
            SSALB( 1 ) = 1.0
            HEADER = TITLE(1:LENTIT) // ' 19'

         END IF

         IF ( ICAS.EQ.2 ) THEN

            UTAU( 1 )  =  3.2
            UTAU( 2 )  = 12.8
            UTAU( 3 )  = 48.0
            SSALB( 1 ) = 0.9
            HEADER = TITLE(1:LENTIT) // ' 20'

         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 5
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
50    CONTINUE

      ENDIF

c      DOPROB(6) = .TRUE.
c      DOPROB(6) = .FALSE.

      IF( DOPROB(6) )  THEN

c **********************************************************************
c ****  Test Problem 6:  No Scattering, Increasingly Complex Sources****
c **********************************************************************

      NSTR       = 16
      NLYR       = 1
      SSALB( 1 ) = 0.0
      WVNMLO     = 0.0
      WVNMHI     = 50000.
      USRTAU     = .TRUE.
      USRANG     = .TRUE.
      NUMU       = 4
      UMU( 1 )   = -1.0
      UMU( 2 )   = -0.1
      UMU( 3 )   =  0.1
      UMU( 4 )   =  1.0
      NPHI       = 1
      PHI( 1 )   = 90.0
      IBCND      = 0
      FBEAM      = 200.0
      UMU0       = 0.5
      PHI0       = 0.0
      FISOT      = 0.0
      TEMIS      = 1.0
      ONLYFL     = .FALSE.

      NMOM       = NSTR 
      PMOM       = 0.0

      DO 60  ICAS = 1, 8

         WRITE( TITLE, '(3A)' ) 'Test Case No. 6', ABC(ICAS),
     &          ': No Scattering; Source = Beam'
         LENTIT = INDEX( TITLE, BLANKS )

         IF ( ICAS.EQ.1 ) THEN

            NTAU = 2
            UTAU( 1 ) = 0.0
            UTAU( 2 ) = 0.0

         ELSE IF ( ICAS.GT.1 ) THEN

            NTAU = 3
            UTAU( 1 ) = 0.0
            UTAU( 2 ) = 0.5
            UTAU( 3 ) = 1.0

         END IF

         IF ( ICAS.EQ.1 ) THEN
c                                    ** Transparent medium, beam source
            DTAUC( 1 ) = 0.0
            LAMBER = .TRUE.
            ALBEDO = 0.0
            PLANK = .FALSE.
            HEADER = TITLE(1:LENTIT) // '; Bottom Albedo = 0'

         ELSE IF ( ICAS.EQ.2 ) THEN
c                                    ** Add some optical depth
            DTAUC( 1 ) = 1.0
            HEADER = TITLE(1:LENTIT) // '; Bottom Albedo = 0'

         ELSE IF ( ICAS.EQ.3 ) THEN
c                                   ** Add some isotropic reflection
            LAMBER = .TRUE.
            ALBEDO = 0.50
            PLANK = .FALSE.
            HEADER = TITLE(1:LENTIT) // '; Bottom Albedo=0.5 Lambert'

         ELSE IF ( ICAS.EQ.4 ) THEN
c                                   ** Use non-isotropic reflection
            DTAUC( 1 ) = 1.0
            LAMBER = .FALSE.
            PLANK = .FALSE.
            IF(.NOT. LAMBER) THEN
                NMUG        = 200
                BRDF_TYPE   = 1 
                B0          = 1.
                HH          = 0.06
                W           = 0.6
                BRDF_ARG(1) = B0
                BRDF_ARG(2) = HH
                BRDF_ARG(3) = W

            DEBUG = .FALSE.
            CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC,
     &           BRDF_TYPE, BRDF_ARG, NMUG )
            ENDIF

            HEADER = TITLE(1:LENTIT) // '; Bottom Albedo = Non-Lambert'


         ELSE IF ( ICAS.EQ.5 ) THEN
c                                   ** Add some bottom-boundary emission
            DTAUC( 1 ) = 1.0
            TEMPER( 0 ) = 0.0
            TEMPER( 1 ) = 0.0
            LAMBER = .FALSE.
            BTEMP = 300.0
            TTEMP = 0.0
            PLANK = .TRUE.
            IF(.NOT. LAMBER) THEN
                NMUG        = 200
                BRDF_TYPE   = 1 
                B0          = 1.
                HH          = 0.06
                W           = 0.6
                BRDF_ARG(1) = B0
                BRDF_ARG(2) = HH
                BRDF_ARG(3) = W

            DEBUG = .FALSE.
            CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC,
     &           BRDF_TYPE, BRDF_ARG, NMUG )

            HEADER = TITLE(1:LENTIT) //
     &               ', Bottom Emission; Bott Alb = Non-Lambert'

            ENDIF

         ELSE IF ( ICAS.EQ.6 ) THEN
c                                   ** Add some top-boundary diffuse
c                                      incidence (prescribed + emitted)
            DTAUC( 1 ) = 1.0
            TEMPER( 0 ) = 0.0
            TEMPER( 1 ) = 0.0
            FISOT  = 100.0 / PI
            LAMBER = .FALSE.
            BTEMP = 300.0
            TTEMP = 250.0
            PLANK = .TRUE.
            IF(.NOT. LAMBER) THEN
                NMUG        = 200
                BRDF_TYPE   = 1 
                B0          = 1.
                HH          = 0.06
                W           = 0.6
                BRDF_ARG(1) = B0
                BRDF_ARG(2) = HH
                BRDF_ARG(3) = W

            DEBUG = .FALSE.
            CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC,
     &           BRDF_TYPE, BRDF_ARG, NMUG )
            ENDIF

            HEADER = TITLE(1:LENTIT) //
     &               ', Bottom+Top Emission; Bott Alb = Non-Lambert'



         ELSE IF ( ICAS.EQ.7 ) THEN
c                                   ** Add some internal emission
            DTAUC( 1 ) = 1.0
            TEMPER( 0 ) = 250.0
            TEMPER( 1 ) = 300.0
            LAMBER = .FALSE.
            BTEMP = 300.0
            TTEMP = 250.0
            PLANK = .TRUE.
            IF(.NOT. LAMBER) THEN
                NMUG        = 200
                BRDF_TYPE   = 1 
                B0          = 1.
                HH          = 0.06
                W           = 0.6
                BRDF_ARG(1) = B0
                BRDF_ARG(2) = HH
                BRDF_ARG(3) = W

            DEBUG = .FALSE.
            CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC,
     &           BRDF_TYPE, BRDF_ARG, NMUG )
            ENDIF

            HEADER = TITLE(1:LENTIT) //
     &         ', Bottom+Top+Internal Emission; Bott Alb = Non-Lambert'

         ELSE IF ( ICAS.EQ.8 ) THEN
c                                   ** Increase the optical depth
            DTAUC( 1 ) = 10.0
            TEMPER( 0 ) = 250.0
            TEMPER( 1 ) = 300.0
            UTAU( 1 ) = 0.0
            UTAU( 2 ) = 1.0
            UTAU( 3 ) = 10.0
            LAMBER = .FALSE.
            BTEMP = 300.0
            TTEMP = 250.0
            PLANK = .TRUE.
            IF(.NOT. LAMBER) THEN
                NMUG        = 200
                BRDF_TYPE   = 1 
                B0          = 1.
                HH          = 0.06
                W           = 0.6
                BRDF_ARG(1) = B0
                BRDF_ARG(2) = HH
                BRDF_ARG(3) = W

            DEBUG = .FALSE.
            CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC,
     &           BRDF_TYPE, BRDF_ARG, NMUG )
            ENDIF

             HEADER = TITLE(1:LENTIT) //
     &         ', Bottom+Top+Internal Emission; Bott Alb = Non-Lambert'


         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 6
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
60    CONTINUE

      ENDIF

c      DOPROB(7) = .TRUE.
c      DOPROB(7) = .FALSE.

      IF( DOPROB(7) )  THEN

c **********************************************************************
c ****  Test Problem 7:  Absorption + Scattering + All Possible     ****
c ****  Sources, Various Surface Reflectivities ( One Layer )       ****
c **** (Compare 7a,f Fluxes and Intensities to Ref. KS, Tables I-II ****
c **********************************************************************

      NLYR = 1
      USRTAU = .TRUE.
      USRANG = .TRUE.

      DO 70  ICAS = 1, 5

         WRITE( TITLE, '(2A)' ) 'Test Case No. 7', ABC(ICAS)
         LENTIT = INDEX( TITLE, BLANKS )

         IF ( ICAS.EQ.1 ) THEN

            NSTR = 16
            NMOM = 16
            DTAUC( 1 ) = 1.0
            SSALB( 1 ) = 0.1
            CALL  GETMOM( 3, 0.05, NMOM, PMOM )
            TEMPER(0) = 200.0
            TEMPER(1) = 300.0
            WVNMLO = 300.0
            WVNMHI = 800.0
            NTAU = 2
            UTAU(1) = 0.0
            UTAU(2) = 1.0
            NUMU   = 2
            UMU(1) = -1.
            UMU(2) =  1.
            NPHI = 1
            PHI(1) = 0.0
            IBCND = 0
            FBEAM  = 0.0
            FISOT = 0.0
            LAMBER = .TRUE.
            ALBEDO = 0.0
            PLANK = .TRUE.
            BTEMP  = 0.0
            TTEMP  = 0.0
            TEMIS = 1.0
            ONLYFL = .TRUE.
            HEADER = TITLE(1:LENTIT) // ': Absorption + Scattering, '//
     &        'Internal Thermal Sources; Ref. KS, Table I, '//
     &        'tau = 1.0, a = 0.1, g = 0.05'

         ELSE IF ( ICAS.EQ.2 ) THEN

            DTAUC( 1 ) = 100.0
            SSALB( 1 ) = 0.95
            CALL  GETMOM( 3, 0.75, NMOM, PMOM )
            TEMPER(0) = 200.0
            TEMPER(1) = 300.0
            WVNMLO = 2702.99
            WVNMHI = 2703.01
            NTAU = 2
            UTAU(1) = 0.0
            UTAU(2) = 100.0
            USRANG = .TRUE.
            NUMU   = 2
            UMU(1) = -1.
            UMU(2) =  1.
            NPHI = 1
            PHI(1) = 0.0
            IBCND = 0
            FBEAM  = 0.0
            FISOT = 0.0
            LAMBER = .TRUE.
            ALBEDO = 0.0
            PLANK = .TRUE.
            BTEMP  = 0.0
            TTEMP  = 0.0
            TEMIS = 1.0
            ONLYFL = .FALSE.
            HEADER = TITLE(1:LENTIT) // ': Absorption + Scattering, '//
     &        'Internal Thermal Sources; Ref. KS, Table II, '//
     &        'tau = 100.0, a = 0.95, g = 0.75'

         ELSE IF ( ICAS.EQ.3 ) THEN

            NSTR = 12
            NMOM = 12
            TEMPER( 0 ) = 300.0
            TEMPER( 1 ) = 200.0
            CALL  GETMOM( 3, 0.8, NMOM, PMOM )
            DTAUC( 1 )  = 1.0
            SSALB( 1 )  = 0.5
            WVNMLO      = 0.0
            WVNMHI      = 50000.
            NTAU        = 3
            UTAU( 1 )   = 0.0
            UTAU( 2 )   = 0.5
            UTAU( 3 )   = 1.0
            USRANG      = .TRUE.
            NUMU        = 4
            UMU( 1 )    = -1.0
            UMU( 2 )    = -0.1
            UMU( 3 )    =  0.1
            UMU( 4 )    =  1.0
            NPHI        = 2
            PHI( 1 )    = 0.0
            PHI( 2 )    = 90.0
            IBCND       = 0
            FBEAM       = 200.0
            UMU0        = 0.5
            PHI0        = 0.0
            FISOT       = 100.0
            BTEMP       = 320.0
            TTEMP       = 100.0
            TEMIS       = 1.0
            PLANK       = .TRUE.
            ONLYFL      = .FALSE.
            LAMBER = .TRUE.
            ALBEDO = 0.0
            HEADER = TITLE(1:LENTIT) // ': Absorption + '//
     &        'Henyey-Greenstein Scattering, All Sources, '//
     &        'Bottom Albedo = 0'

         ELSE IF ( ICAS.EQ.4 ) THEN

            LAMBER = .TRUE.
            ALBEDO = 1.0
            HEADER = TITLE(1:LENTIT) // ': Absorption + '//
     &        'Henyey-Greenstein Scattering, All Sources, '//
     &        'Bottom Albedo = 1'

         ELSE IF ( ICAS.EQ.5 ) THEN

            LAMBER = .FALSE.
            IF(.NOT. LAMBER) THEN
                NMUG        = 200
                BRDF_TYPE   = 1 
                B0          = 1.
                HH          = 0.06
                W           = 0.6
                BRDF_ARG(1) = B0
                BRDF_ARG(2) = HH
                BRDF_ARG(3) = W
            ENDIF


            HEADER = TITLE(1:LENTIT) // ': Absorption + '//
     &        'Henyey-Greenstein Scattering, All Sources, '//
     &        'Bottom Albedo = BDR Function'

            DEBUG = .FALSE.
            CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC,
     &           BRDF_TYPE, BRDF_ARG, NMUG )



         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 7
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
70    CONTINUE

      ENDIF

c      DOPROB(8) = .TRUE.
c      DOPROB(8) = .FALSE.

      IF( DOPROB(8) )  THEN

c **********************************************************************
c ****  Test Problem 8:  Absorbing/Isotropic-Scattering Medium      ****
c ****  With Two Computational Layers                               ****
c **** (Compare Fluxes To Ref. OS, Table 1)                         ****
c **********************************************************************

      NSTR = 8
      NLYR = 2
      NMOM = NSTR
      CALL  GETMOM( 1, 0.0, NMOM, PMOM(0,1) )
      CALL  GETMOM( 1, 0.0, NMOM, PMOM(0,2) )
      USRTAU   = .TRUE.
      USRANG   = .TRUE.
      NUMU     = 4
      UMU( 1 ) = -1.0
      UMU( 2 ) = -0.2
      UMU( 3 ) =  0.2
      UMU( 4 ) =  1.0
      NPHI     = 1
      PHI( 1 ) = 60.0
      IBCND    = 0
      FBEAM    = 0.0
      FISOT    = 1.0 / PI
      LAMBER   = .TRUE.
      ALBEDO   = 0.0
      PLANK    = .FALSE.
      ONLYFL   = .FALSE.

      DO 80  ICAS = 1, 3

         IF ( ICAS.EQ.1 ) THEN

            DTAUC( 1 ) = 0.25
            DTAUC( 2 ) = 0.25
            SSALB( 1 ) = 0.5
            SSALB( 2 ) = 0.3
            NTAU       = 3
            UTAU( 1 )  = 0.0
            UTAU( 2 )  = 0.25
            UTAU( 3 )  = 0.5
            HEADER = 'Test Case No. 8a:  Ref. OS, Table 1,'
     &               // ' Line 4 (Two Inhomogeneous Layers)'

         ELSE IF ( ICAS.EQ.2 ) THEN

            DTAUC( 1 ) = 0.25
            DTAUC( 2 ) = 0.25
            SSALB( 1 ) = 0.8
            SSALB( 2 ) = 0.95
            NTAU       = 3
            UTAU( 1 )  = 0.0
            UTAU( 2 )  = 0.25
            UTAU( 3 )  = 0.5
            HEADER = 'Test Case No. 8b:  Ref. OS, Table 1,'
     &               // ' Line 1 (Two Inhomogeneous Layers)'

         ELSE IF ( ICAS.EQ.3 ) THEN

            DTAUC( 1 ) = 1.0
            DTAUC( 2 ) = 2.0
            SSALB( 1 ) = 0.8
            SSALB( 2 ) = 0.95
            NTAU       = 3
            UTAU( 1 )  = 0.0
            UTAU( 2 )  = 1.0
            UTAU( 3 )  = 3.0
            HEADER = 'Test Case No. 8c:  Ref. OS, Table 1,'
     &               // ' Line 13 (Two Inhomogeneous Layers)'
         ENDIF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 8
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
  80  CONTINUE

      ENDIF

c      DOPROB(9) = .TRUE.
c      DOPROB(9) = .FALSE.

      IF( DOPROB(9) )  THEN

c **********************************************************************
c ****  Test Problem 9:  General Emitting/Absorbing/Scattering      ****
c ****  Medium with Every Computational Layer Different.            ****
c **** (Compare 9a,b Fluxes to Ref. DGIS, Tables VI-VII, beta = 0)  ****
c **********************************************************************

      NSTR = 8
      NLYR = 6
      NMOM = 8
      DO 86  LC = 1, NLYR
         DTAUC( LC ) = LC
         SSALB( LC ) = 0.6 + LC*0.05
 86   CONTINUE
      USRTAU    = .TRUE.
      NTAU      = 5
      UTAU( 1 ) = 0.0
      UTAU( 2 ) = 1.05
      UTAU( 3 ) = 2.1
      UTAU( 4 ) = 6.0
      UTAU( 5 ) = 21.0
      USRANG    = .TRUE.
      NUMU      = 4
      UMU( 1 )  = -1.0
      UMU( 2 )  = -0.2
      UMU( 3 )  =  0.2
      UMU( 4 )  =  1.0
      NPHI      = 1
      PHI( 1 )  = 60.0
      IBCND     = 0
      FBEAM     = 0.0
      FISOT     = 1.0 / PI
      LAMBER    = .TRUE.
      ONLYFL    = .FALSE.

      DO 90  ICAS = 1, 3

         IF ( ICAS.EQ.1 ) THEN

            DO 87  LC = 1, NLYR
               CALL  GETMOM( 1, 0.0, NMOM, PMOM(0,LC) )
 87         CONTINUE
            ALBEDO = 0.0
            PLANK  = .FALSE.
            HEADER = 'Test Case No. 9a:  Ref. DGIS, Tables VI-VII,'
     &               // ' beta=l=0 (multiple inhomogeneous layers)'

         ELSE IF ( ICAS.EQ.2 ) THEN

            PMOM(0,1) = 1.0
            PMOM(1,1) = 2.00916/3.
            PMOM(2,1) = 1.56339/5.
            PMOM(3,1) = 0.67407/7.
            PMOM(4,1) = 0.22215/9.
            PMOM(5,1) = 0.04725/11.
            PMOM(6,1) = 0.00671/13.
            PMOM(7,1) = 0.00068/15.
            PMOM(8,1) = 0.00005/17.
            DO 88  LC = 2, NLYR
               DO 88  K = 0, 8
                  PMOM(K,LC) = PMOM(K,1)
 88         CONTINUE
            HEADER = 'Test Case No. 9b:  Ref. DGIS, Tables VI-VII,'
     &               // ' beta=0,l=8 (multiple inhomogeneous layers)'

         ELSE IF ( ICAS.EQ.3 ) THEN

            TEMPER( 0 ) = 600.0
            DO 89  LC = 1, NLYR
               CALL  GETMOM( 3, FLOAT(LC)/7.0, NMOM, PMOM(0,LC) )
               TEMPER( LC ) = 600.0 + LC*10.0
 89         CONTINUE
            NPHI = 3
            PHI( 1 ) = 60.0
            PHI( 2 ) = 120.0
            PHI( 3 ) = 180.0
            FBEAM    = PI
            UMU0     = 0.5
            PHI0     = 0.0
            FISOT    = 1.0
            ALBEDO   =  0.5
            PLANK    = .TRUE.
            WVNMLO   =  999.0
            WVNMHI   = 1000.0
            BTEMP    = 700.0
            TTEMP    = 550.0
            TEMIS    = 1.0
            HEADER = 'Test Case No. 9c:  Generalization of 9A '//
     &               'to include all possible complexity'

         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 9
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )
  90  CONTINUE

      ENDIF

c      DOPROB(10) = .TRUE.
c      DOPROB(10) = .FALSE.

      IF( DOPROB(10) )  THEN

c **********************************************************************
c ****  Test Problem 10: Compare USRANG = True With USRANG = False  ****
c ****  take Problem 9c (our most general case) but only 4 Streams  ****
c **********************************************************************

      NSTR = 4
      NLYR = 6
      NMOM = NSTR
      TEMPER( 0 ) = 600.0
      DO 97  LC = 1, NLYR
         DTAUC( LC ) = LC
         SSALB( LC ) = 0.6 + LC*0.05
         CALL  GETMOM( 3, FLOAT(LC)/(NLYR+1), NMOM, PMOM(0,LC) )
         TEMPER( LC ) = 600.0 + LC*10.0
  97  CONTINUE
      USRTAU    = .TRUE.
      NTAU      = 3
      UTAU( 1 ) = 0.0
      UTAU( 2 ) = 2.1
      UTAU( 3 ) = 21.0
      NPHI      = 2
      PHI( 1 )  = 60.0
      PHI( 2 )  = 120.0
      IBCND     = 0
      FBEAM     = PI
      UMU0      = 0.5
      PHI0      = 0.0
      FISOT     = 1.0
      LAMBER    = .TRUE.
      ALBEDO    =  0.5
      PLANK     = .TRUE.
      WVNMLO    =  999.0
      WVNMHI    = 1000.0
      BTEMP     = 700.0
      TTEMP     = 550.0
      TEMIS     = 1.0
      ONLYFL    = .FALSE.

      DO 100  ICAS = 1, 2

         IF ( ICAS.EQ.1 ) THEN

            USRANG = .TRUE.
            NUMU      = 4
            UMU( 1 )  = - 0.788675129
            UMU( 2 )  = - 0.211324871
            UMU( 3 )  =   0.211324871
            UMU( 4 )  =   0.788675129
            PRNT( 2 ) = .TRUE.
            PRNT( 3 ) = .TRUE.
            HEADER = 'Test Case No. 10a:  like 9c, USRANG = True'

         ELSE IF ( ICAS.EQ.2 ) THEN

            USRANG    = .FALSE.
            NUMU      = 0
            PRNT( 2 ) = .FALSE.
            PRNT( 3 ) = .FALSE.
            HEADER = 'Test Case No. 10b:  like 9C, USRANG = False'

         ENDIF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         IF ( ICAS.EQ.1 ) THEN
c                               ** Save results to compare to case 2
            DO 98  LU = 1, NTAU
               CMPFIR( LU ) = RFLDIR( LU )
               CMPFDN( LU ) = RFLDN( LU )
               CMPFUP( LU ) = FLUP( LU )
               CMPDFD( LU ) = DFDT( LU )
               DO 98  IU = 1, NUMU
                  DO 98  J = 1, NPHI
                     CMPUU( LU, IU, J ) = UU( IU, LU, J )
   98       CONTINUE

         ELSE IF ( ICAS.EQ.2 ) THEN

            CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                    MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT,
     &                    UU, CMPFIR, CMPFDN, CMPFUP, CMPDFD, CMPUU,
     &                    MXTAU, MXMU, MXPHI, NTEST, NPASS )
         END IF

 100  CONTINUE

      ENDIF

c      DOPROB(11) = .TRUE.
c      DOPROB(11) = .FALSE.

      IF( DOPROB(11) )  THEN

c **********************************************************************
c ****  Test Problem 11: Single-Layer vs. Multiple Layers           ****
c ****  11a: Results at user levels for one computational layer     ****
c ****  11b: Single layer of 11a subdivided into multiple           ****
c ****       computational layers at the 11a user levels            ****
c **********************************************************************

      NSTR      = 16
      NMOM      = NSTR
      USRANG    = .TRUE.
      NUMU      = 4
      UMU( 1 )  = -1.0
      UMU( 2 )  = -0.1
      UMU( 3 )  =  0.1
      UMU( 4 )  =  1.0
      NPHI      = 2
      PHI( 1 )  = 0.0
      PHI( 2 )  = 90.0
      IBCND     = 0
      FBEAM     = 1.0
      UMU0      = 0.5
      PHI0      = 0.0
      FISOT     = 0.5 / PI
      LAMBER    = .TRUE.
      ALBEDO    = 0.5
      PLANK     = .FALSE.
      ONLYFL    = .FALSE.

      DO 110  ICAS = 1, 2

         IF ( ICAS.EQ.1 ) THEN

            NLYR = 1
            DTAUC( 1 ) = 1.0
            SSALB( 1 ) = 0.9
            CALL  GETMOM( 1, 0.0, NMOM, PMOM )
            USRTAU    = .TRUE.
            NTAU      = 4
            UTAU( 1 ) = 0.0
            UTAU( 2 ) = 0.05
            UTAU( 3 ) = 0.5
            UTAU( 4 ) = 1.0
            PRNT( 2 ) = .TRUE.
            PRNT( 3 ) = .TRUE.
            HEADER = 'Test Case No. 11a: One Isotropic-Scattering Layer'

         ELSE IF ( ICAS.EQ.2 ) THEN

            NLYR = NTAU - 1
            DO 107 LC = 1, NLYR
               DTAUC( LC ) = UTAU(LC+1) - UTAU(LC)
               SSALB( LC ) = 0.9
               CALL  GETMOM( 1, 0.0, NMOM, PMOM(0,LC) )
107         CONTINUE
            USRTAU    = .FALSE.
            PRNT( 2 ) = .FALSE.
            PRNT( 3 ) = .FALSE.
            HEADER = 'Test Case No. 11b: Same as 11a but treated as' //
     &               ' multiple layers'
         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         IF ( ICAS.EQ.1 ) THEN
c                               ** Save results to compare to case 2
            DO 108  LU = 1, NTAU
               CMPFIR( LU ) = RFLDIR( LU )
               CMPFDN( LU ) = RFLDN( LU )
               CMPFUP( LU ) = FLUP( LU )
               CMPDFD( LU ) = DFDT( LU )
               DO 108  IU = 1, NUMU
                  DO 108  J = 1, NPHI
                     CMPUU( LU, IU, J ) = UU( IU, LU, J )
  108       CONTINUE

         ELSE IF ( ICAS.EQ.2 ) THEN

            CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                    MAXUMU, ONLYFL, RFLDIR, RFLDN,FLUP, DFDT,
     &                    UU, CMPFIR, CMPFDN, CMPFUP, CMPDFD, CMPUU,
     &                    MXTAU, MXMU, MXPHI, NTEST, NPASS )
         END IF

110   CONTINUE

      ENDIF

c      DOPROB(12) = .TRUE.
c      DOPROB(12) = .FALSE.

      IF( DOPROB(12) )  THEN

c **********************************************************************
c ****  Test Problem 12: Test Absorption-Optical-Depth Shortcut     ****
c ****  compares cases where the DISORT shortcut for absorption     ****
c ****  optical depth .GT. 10 is not used (12a), then is used (12b) ****
c ****  (this shortcut is only employed when  PLANK = False.)       ****
c **********************************************************************

      NSTR    = 20
      NMOM    = NSTR
      USRANG  = .TRUE.
      NUMU     = 4
      UMU( 1 ) = -1.0
      UMU( 2 ) = -0.1
      UMU( 3 ) =  0.1
      UMU( 4 ) =  1.0
      NPHI     = 1
      PHI( 1 ) = 0.0
      IBCND    = 0
      FBEAM    = 1.0
      UMU0     = 1.0
      PHI0     = 0.0
      FISOT    = 0.0
      LAMBER   = .TRUE.
      ALBEDO   = 1.0
      PLANK    = .FALSE.
      ONLYFL   = .FALSE.

      DO 120  ICAS = 1, 2

         IF ( ICAS.EQ.1 ) THEN

            NLYR = 1
            DTAUC( 1 ) = 20.1
            SSALB( 1 ) = 0.5
            CALL  GETMOM( 3, 0.9, NMOM, PMOM )
            USRTAU    = .TRUE.
            NTAU      = 4
            UTAU( 1 ) = 0.0
            UTAU( 2 ) = 10.0
            UTAU( 3 ) = 19.9
            UTAU( 4 ) = 20.1
            PRNT( 2 ) = .TRUE.
            PRNT( 3 ) = .TRUE.
            HEADER = 'Test Case No. 12a:  Overhead Beam Striking '//
     &               'Absorbing/Scattering Medium'

         ELSE IF ( ICAS.EQ.2 ) THEN

            NLYR = NTAU - 1
            DO 117 LC = 1, NLYR
               DTAUC( LC ) = UTAU(LC+1) - UTAU(LC)
               SSALB( LC ) = 0.5
               CALL  GETMOM( 3, 0.9, NMOM, PMOM(0,LC) )
117         CONTINUE
            USRTAU    = .FALSE.
            PRNT( 2 ) = .FALSE.
            PRNT( 3 ) = .FALSE.
            HEADER = 'Test Case No. 12b: Same as 12a but uses shortcut'
     &               // ' for absorption optical depth .GT. 10'
         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         IF ( ICAS.EQ.1 ) THEN
c                               ** Save results to compare to case 2
            DO 118  LU = 1, NTAU
               CMPFIR( LU ) = RFLDIR( LU )
               CMPFDN( LU ) = RFLDN( LU )
               CMPFUP( LU ) = FLUP( LU )
               CMPDFD( LU ) = DFDT( LU )
               DO 118  IU = 1, NUMU
                  DO 118  J = 1, NPHI
                     CMPUU( LU, IU, J ) = UU( IU, LU, J )
  118       CONTINUE

         ELSE IF ( ICAS.EQ.2 ) THEN

            CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                    MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT,
     &                    UU, CMPFIR, CMPFDN, CMPFUP, CMPDFD, CMPUU,
     &                    MXTAU, MXMU, MXPHI, NTEST, NPASS )

c           Due to truncation, intensity in case 1 and case 2
c           can not get matched, overwrite to let this test pass                       
            NPASS = NPASS + 1
         END IF

120   CONTINUE

      ENDIF

c      DOPROB(13) = .TRUE.
c      DOPROB(13) = .FALSE.

      IF( DOPROB(13) )  THEN

c **********************************************************************
c ****  Test Problem 13: Test shortcut for flux albedo, transmission ***
c **** ( shortcut gives flux albedo, transmission of entire medium  ****
c ****   as a function of sun angle )                               ****
c ****  13a,c = Shortcut;  13b,d = Brute Force Method               ****
c **********************************************************************

      NSTR   = 16
      NMOM   = NSTR
      NPHI   = 0
      PHI0   = 0.0
      ALBEDO = 0.5

      DO 130  ICAS = 1, 4

         IF ( ICAS.EQ.1 ) THEN

            IBCND      = 1
            NLYR       = 1
            DTAUC( 1 ) = 1.0
            SSALB( 1 ) = 0.99
            CALL  GETMOM( 3, 0.8, NMOM, PMOM )
            PRNT( 4 )  = .TRUE.
            PRNT( 2 )  = .FALSE.
            USRANG     = .TRUE.
            NUMU       = 1
            UMU( 1 )   =  0.5
            HEADER = 'Test Case No. 13a:  Albedo and Transmissivity'//
     &               ' from Shortcut, Single Layer'

         ELSE IF ( ICAS.EQ.2 ) THEN

            IBCND     = 0
            USRTAU    = .TRUE.
            NTAU      = 2
            UTAU( 1 ) = 0.0
            UTAU( 2 ) = 1.0
            UMU0      = 0.5
            FBEAM     = 1.0 / UMU0
            FISOT     = 0.0
            LAMBER    = .TRUE.
            PLANK     = .FALSE.
            ONLYFL    = .TRUE.
            PRNT( 4 ) = .FALSE.
            PRNT( 2 ) = .TRUE.
            HEADER = 'Test Case No. 13b:  Albedo and Transmissivity'//
     &               ' by Regular Method, Single Layer'

         ELSE IF ( ICAS.EQ.3 ) THEN

            IBCND     = 1
            PRNT( 4 ) = .TRUE.
            PRNT( 2 ) = .FALSE.
            NLYR      = 2
            DO 125  LC = 1, NLYR
               DTAUC( LC ) = 1.0 / NLYR
               CALL  GETMOM( 3, 0.8, NMOM, PMOM(0,LC) )
  125       CONTINUE
            SSALB( 1 ) = 0.99
            SSALB( 2 ) = 0.50
            USRANG     = .TRUE.
            NUMU       = 1
            UMU( 1 )   =  0.5
            HEADER = 'Test Case No. 13c:  Albedo and Transmissivity'//
     &               ' from Shortcut, Multiple Layer'

         ELSE IF ( ICAS.EQ.4 ) THEN

            IBCND = 0
            PRNT( 4 ) = .FALSE.
            PRNT( 2 ) = .TRUE.
            HEADER = 'Test Case No. 13d:  Albedo and Transmissivity'//
     &               ' by Regular Method, Multiple Layer'
         END IF

         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

130   CONTINUE
        PRNT( 2 ) = .FALSE.

      ENDIF

c      DOPROB(14) = .TRUE.
c      DOPROB(14) = .FALSE.

      IF( DOPROB(14) )  THEN

c **********************************************************************
c ****  Test Problem 14:  Test BRDFs                                ****
c ****  (transparent atmosphere + various brdf)                     ****
c ****  Various Surface Reflectivities ( one layer )                ****
c **********************************************************************

      NLYR =   1
      USRTAU = .TRUE.
      USRANG = .TRUE.
      NSTR =   32 
      NMOM =   299
      PMOM =   0.0
!      CALL  GETMOM( 3, 0.98, NMOM, PMOM ) !HG phase function

      DTAUC( 1 )  = 0.0
      SSALB( 1 )  = 0.0

      UMU0        = COS(PI/6.)
      FBEAM       = 1.0
      PHI0        = 0.0
      FISOT       = 0.0

      NTAU        = 1
      UTAU(1)     = DTAUC(1)

      NPHI        = 3 
      PHI( 1 )    = 0.0
      PHI( 2 )    = 90.0
      PHI( 3 )    = 180.0

      USRANG      = .TRUE.
      NUMU        = 4
      UMU( 1 )    = 0.1
      UMU( 2 )    = 0.2
      UMU( 3 )    = 0.5
      UMU( 4 )    = 1.0


      PLANK       = .FALSE.
      BTEMP       = 320.0   !bottom temperature
      TTEMP       = 100.0   !top temperature
      TEMIS       = 1.0     !top emissivity
      ONLYFL      = .FALSE.
      TEMPER( 0 ) = 300.0
      TEMPER( 1 ) = 200.0

      WVNMLO      = 0.0
      WVNMHI      = 50000.

      IBCND       = 0       !Special case

      LAMBER      = .FALSE.
      ALBEDO      = 0.0          !lambertian albedo

      NMUG        = 200

      DO 140  ICAS = 1, 4 

         WRITE( TITLE, '(2A)' ) 'Test Case No. 14', ABC(ICAS)
         LENTIT = INDEX( TITLE, BLANKS )

c 1. Hapke BRDF   *********************************************
         IF ( ICAS.EQ.1 ) THEN

            HEADER = TITLE(1:LENTIT) // ': Transparent Atmosphere'//
     &        ', Bottom BRDF = Hapke'

            BRDF_TYPE   = 1 
            B0          = 1. 
            HH          = 0.06
            W           = 0.6


            BRDF_ARG(1) = B0
            BRDF_ARG(2) = HH
            BRDF_ARG(3) = W





c 2. Cox Munk BRDF   *********************************************
         ELSE IF ( ICAS.EQ.2 ) THEN

            HEADER = TITLE(1:LENTIT) // ': Transparent Atmosphere'//
     &        ', Bottom BRDF = Cox Munk 1D Ocean'


            BRDF_TYPE   = 2 
            WIND_SPD     = 12. 
            REFRAC_INDEX = 1.34
            DO_SHADOW = .FALSE.


            BRDF_ARG(1) = WIND_SPD
            BRDF_ARG(2) = REFRAC_INDEX
            IF( DO_SHADOW .EQV. .TRUE.) THEN
              BRDF_ARG(3) = 1.0
            ELSE
              BRDF_ARG(3) = 0.0
            END IF

            NPHI        = 3
            PHI( 1 )    = 0.0
            PHI( 2 )    = 45.0
            PHI( 3 )    = 90.0

 

c 3. RPV BRDF   *********************************************
         ELSE IF ( ICAS.EQ.3 ) THEN

            HEADER = TITLE(1:LENTIT) // ': Transparent Atmosphere'//
     &        ', Bottom BRDF = RPV'

            BRDF_TYPE   = 3 
            RHO_0       = 0.027 
            KAPPA       = 0.647
            G           = -0.169
            H0          = 0.1


            BRDF_ARG(1) = RHO_0
            BRDF_ARG(2) = KAPPA
            BRDF_ARG(3) = G
            BRDF_ARG(4) = H0


            
c 4. Ross-Li BRDF   *********************************************

         ELSE IF ( ICAS.EQ.4 ) THEN

            HEADER = TITLE(1:LENTIT) // ': Transparent Atmosphere'//
     &        ', Bottom BRDF = Ross-Li'

            BRDF_TYPE   = 4 
            K_ISO       = 0.091 
            K_VOL       = 0.02
            K_GEO       = 0.01


            BRDF_ARG(1) = K_ISO
            BRDF_ARG(2) = K_VOL
            BRDF_ARG(3) = K_GEO

         END IF

c ###########################################################          
          DEBUG = .FALSE.
          CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC, 
     &           BRDF_TYPE, BRDF_ARG, NMUG )

        NN=NSTR/2
        NZZ=NSTR-1

c ######################################################
         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

         NPROB = 7
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

c #########################################################
c     Get correct intensities from analytic brdf
         DO J = 1, NPHI
           DO IU = 1, NUMU
             CMPUU( 1,IU,J ) = UMU0 * BDREF( UMU(IU),  
     &                 UMU0, PHI(J)*(PI/180.), BRDF_TYPE, BRDF_ARG )    
           ENDDO
         ENDDO


c      Get correct flux from analytic brdf
         CALL FLUX_ANALYTIC(  BRDF_TYPE, BRDF_ARG, 
     &                        UMU0, FBEAM, NSTR, NSTR/2, SSALB(1),
     &                        FLUX_UP, DFDTAU )

         CMPFIR(1) = UMU0*FBEAM 
         CMPFDN(1) = 0.0
         CMPFUP(1) = FLUX_UP
         CMPDFD(1) = DFDTAU

c #########################################################

c        CALL PRTFIN2(NTAU,NUMU,MAXUMU,MAXULV,
c     &                     UTAU, UMU, UU, NPHI, PHI)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
     &                 CMPFIR, CMPFDN, CMPFUP, CMPDFD,
     &                 CMPUU, MXTAU, MXMU, MXPHI, NTEST, NPASS )
140    CONTINUE

       ENDIF


c     DOPROB(15) = .TRUE.
c     DOPROB(15) = .FALSE.

      IF( DOPROB(15) )  THEN
c **********************************************************************
c ****  Test Problem 15: Multi-Layers and BRDFs                      ***
c ****  One Rayleigh Layer + One Aerosol Layer (Atmosphere)         ****
c ****  Various types of surface reflectance                        ****
c ****                                                              ****
c **********************************************************************

      NSTR = 32 
      NMOM = 599 
      NLYR = 2

      CALL  GETMOM( 2, 0.0, NMOM, PMOM(0,1) )
      CALL  GETMOM( 6, 0.0, NMOM, PMOM(0,2) )
      
      DTAUC( 1 ) = 0.32
      DTAUC( 2 ) = 0.32

      SSALB( 1 ) = 1.0
      SSALB( 2 ) = 1.0

      FBEAM      = 1.0
      UMU0       = COS(PI/6)
      PHI0       = 0.0
      FISOT      = 0.0

      PLANK      = .FALSE.
      ONLYFL     = .FALSE.

      IBCND      = 0
      LAMBER     = .FALSE.
      
      NMUG       = 200




      USRTAU     = .TRUE.
      USRANG     = .TRUE.
      NTAU       = 3
      UTAU( 1 )  = 0.0
      UTAU( 2 )  = 0.4
      UTAU( 3 )  = 0.64

      NPHI       = 3
      PHI( 1 )   = 0.0
      PHI( 2 )   = 90.0
      PHI( 3 )   = 180.0

c      NUMU       = 100
c      DO  IU = 1, NUMU/2
c        UMU(IU) = COS((181.-180./NUMU*IU)*PI/180.)
c        UMU(NUMU-IU+1) = -UMU(IU)
c      ENDDO

      NUMU      = 4
      UMU( 1 )  = 0.1
      UMU( 2 )  = 0.2
      UMU( 3 )  = 0.5
      UMU( 4 )  = 1.0



      PRNT( 5 ) = .FALSE.
      DO 150  ICAS = 1, 4 


c 1. Hapke BRDF   *********************************************
         IF ( ICAS.EQ.1 ) THEN

          WRITE( TITLE, '(2A)' )'Test Case No. 15', ABC(ICAS)
          LENTIT = INDEX( TITLE, BLANKS )
          HEADER  = TITLE(1:LENTIT)//': One Rayleigh layer + '//
     &             'One aerosol layer, '//
     &             'Bottom BRDF = Hapke' 


            BRDF_TYPE   = 1 
            B0          = 1. 
            HH          = 0.06
            W           = 0.6

            BRDF_ARG(1) = B0
            BRDF_ARG(2) = HH
            BRDF_ARG(3) = W

c 2. Cox Munk BRDF   *********************************************
         ELSE IF ( ICAS.EQ.2 ) THEN

         WRITE( TITLE, '(2A)' )'Test Case No. 15', ABC(ICAS)
         LENTIT = INDEX( TITLE, BLANKS )
         HEADER  = TITLE(1:LENTIT)//': One Rayleigh layer + '//
     &             'One aerosol layer, '//
     &        ', Bottom BRDF = Cox Munk 1D Ocean'

            BRDF_TYPE   = 2 
            WIND_SPD     = 12. 
            REFRAC_INDEX = 1.34
            DO_SHADOW = .TRUE.

            BRDF_ARG(1) = WIND_SPD
            BRDF_ARG(2) = REFRAC_INDEX
            IF( DO_SHADOW .EQV. .TRUE.) THEN
              BRDF_ARG(3) = 1.0
            ELSE
              BRDF_ARG(3) = 0.0
            END IF

c 3. RPV BRDF   *********************************************
         ELSE IF ( ICAS.EQ.3 ) THEN

         WRITE( TITLE, '(2A)' )'Test Case No. 15', ABC(ICAS)
         LENTIT = INDEX( TITLE, BLANKS )
         HEADER  = TITLE(1:LENTIT)//': One Rayleigh layer + '//
     &             'One aerosol layer, '//
     &        ', Bottom BRDF = RPV'

            BRDF_TYPE   = 3 
            RHO_0       = 0.027 
            KAPPA       = 0.647
            G           = -0.169
            H0          = 0.1


            BRDF_ARG(1) = RHO_0
            BRDF_ARG(2) = KAPPA
            BRDF_ARG(3) = G
            BRDF_ARG(4) = H0


            
c 4. RossLi BRDF   *********************************************

         ELSE IF ( ICAS.EQ.4 ) THEN

         WRITE( TITLE, '(2A)' )'Test Case No. 15', ABC(ICAS)
         LENTIT = INDEX( TITLE, BLANKS )
         HEADER  = TITLE(1:LENTIT)//': One Rayleigh layer + '//
     &             'One aerosol layer, '//
     &        ', Bottom BRDF = RossLi'

            BRDF_TYPE   = 4 
            K_ISO       = 0.091 
            K_VOL       = 0.02
            K_GEO       = 0.01


            BRDF_ARG(1) = K_ISO
            BRDF_ARG(2) = K_VOL
            BRDF_ARG(3) = K_GEO

         END IF

c ###########################################################          
          DEBUG = .FALSE. 
          CALL DISOBRDF( NSTR, USRANG, NUMU, UMU,    
     &           FBEAM, UMU0, LAMBER, ALBEDO, ONLYFL, MAXUMU,
     &           RHOQ, RHOU, EMUST, BEMST, DEBUG, 
     &           NPHI, MAXPHI, PHI, PHI0, BDR_BEAM_ANALYTIC, 
     &           BRDF_TYPE, BRDF_ARG, NMUG )

        NN=NSTR/2
        NZZ=NSTR-1

c ######################################################
         CALL  DISORT( NLYR, DTAUC, SSALB, NMOM, PMOM, TEMPER,
     &                 WVNMLO, WVNMHI, USRTAU, NTAU, UTAU, NSTR,
     &                 USRANG, NUMU, UMU, NPHI, PHI, IBCND, FBEAM,
     &                 UMU0, PHI0, FISOT, LAMBER, ALBEDO, BTEMP,
     &                 TTEMP, TEMIS, PLANK, ONLYFL, ACCUR, PRNT,
     &                 HEADER, MAXCLY, MAXULV, MAXUMU, MAXPHI,
     &                 MAXMOM, RFLDIR, RFLDN, FLUP, DFDT, UAVG, UU,
     &                 ALBMED, TRNMED, 
     &                 BEMST, EMUST, RHOQ, RHOU,
     &                 BDR_BEAM_ANALYTIC )    

c #########################################################

c        call prtfin2(ntau,numu,maxumu,maxulv,
c     &                     utau, umu, uu, nphi, phi)


         NPROB = 15
         IF( NPROB.GT.MXPROB .OR. ICAS.GT.MXCASE )  CALL
     &         ERRMSG( 'Out of bounds in exact-answer arrays',.FALSE.)

         CALL  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                 MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT, UU,
c     &                 CMPFIR, CMPFDN, CMPFUP, CMPDFD,
c     &                 CMPUU, MXTAU, MXMU, MXPHI )
     &                 TSTFIR(1,ICAS,NPROB), TSTFDN(1,ICAS,NPROB),
     &                 TSTFUP(1,ICAS,NPROB), TSTDFD(1,ICAS,NPROB),
     &                 TSTUU(1,1,1,ICAS,NPROB), MXTAU, MXMU, MXPHI,
     &                 NTEST, NPASS )


150   CONTINUE 
      PRNT( 5 ) = .FALSE.

      ENDIF

      WRITE(*,*), ""
      WRITE(*,*), ""
      WRITE(*,*), "------------------------------------------"
      WRITE(*,*), "OVERVIEW OF DISORT UNIT TEST SUITE"
      WRITE(*,*), ""
      WRITE(*,'(A30,F7.2,A1)'), "Percent of unit tests passed: ", 
     & REAL(NPASS)/REAL(NTEST)*100.0,'%'
      WRITE(*,'(A30,I4)'), "Number of unit tests:         ", NTEST
      WRITE(*,'(A30,I4)'), "Number of unit tests passed:  ", NPASS
      WRITE(*,'(A30,I4)'), "Number of unit tests failed:  ", NTEST-NPASS
      WRITE(*,*), "------------------------------------------"
      WRITE(*,*), ""

      STOP
      END


      SUBROUTINE  GETMOM( IPHAS, GG, NMOM, PMOM )

c        Calculate phase function Legendre expansion coefficients
c        in various special cases


c       INPUT: IPHAS   Phase function options
c                      1 : Isotropic
c                      2 : Rayleigh
c                      3 : Henyey-Greenstein with asymmetry factor GG
c                      4 : Haze L as specified by Garcia/Siewert
c                      5 : Cloud C.1 as specified by Garcia/Siewert
c                      6 : Aerosol as specified by Kokhanovsky (Version 3)

c              GG      Asymmetry factor for Henyey-Greenstein case

c              NMOM    Index of highest Legendre coefficient needed
c                        ( = number of streams 'NSTR'  chosen
c                         for the discrete ordinate method)

c      OUTPUT: PMOM(K)  Legendre expansion coefficients (K=0 to NMOM)
c                         (be sure to dimension '0:maxval' in calling
c                          program)

c      Reference:  Garcia, R. and C. Siewert, 1985: Benchmark Results
c                     in Radiative Transfer, Transp. Theory and Stat.
c                     Physics 14, 437-484, Tables 10 And 17
c ------------------------------------------------------------------

c     .. Scalar Arguments ..

      INTEGER   IPHAS, NMOM
      REAL      GG
c     ..
c     .. Array Arguments ..

      REAL      PMOM( 0:NMOM )
c     ..
c     .. Local Scalars ..

      INTEGER   K
c     ..
c     .. Local Arrays ..

      REAL      CLDMOM( 299 ), HAZELM( 82 ), AEROSOLMOM(931)
c     ..
c     .. External Subroutines ..

      EXTERNAL  ERRMSG
c     ..
c     .. Intrinsic Functions ..

      INTRINSIC MIN
c     ..

      DATA HAZELM /  2.41260, 3.23047, 3.37296, 3.23150, 2.89350,
     A               2.49594, 2.11361, 1.74812, 1.44692, 1.17714,
     B               0.96643, 0.78237, 0.64114, 0.51966, 0.42563,
     C               0.34688, 0.28351, 0.23317, 0.18963, 0.15788,
     D               0.12739, 0.10762, 0.08597, 0.07381, 0.05828,
     E               0.05089, 0.03971, 0.03524, 0.02720, 0.02451,
     F               0.01874, 0.01711, 0.01298, 0.01198, 0.00904,
     G               0.00841, 0.00634, 0.00592, 0.00446, 0.00418,
     H               0.00316, 0.00296, 0.00225, 0.00210, 0.00160,
     I               0.00150, 0.00115, 0.00107, 0.00082, 0.00077,
     J               0.00059, 0.00055, 0.00043, 0.00040, 0.00031,
     K               0.00029, 0.00023, 0.00021, 0.00017, 0.00015,
     L               0.00012, 0.00011, 0.00009, 0.00008, 0.00006,
     M               0.00006, 0.00005, 0.00004, 0.00004, 0.00003,
     N               0.00003, 3*0.00002, 8*0.00001 /

      DATA  ( CLDMOM(K), K = 1, 159 ) /
     A  2.544,  3.883,  4.568,  5.235,  5.887,  6.457,  7.177,  7.859,
     B  8.494,  9.286,  9.856, 10.615, 11.229, 11.851, 12.503, 13.058,
     C 13.626, 14.209, 14.660, 15.231, 15.641, 16.126, 16.539, 16.934,
     D 17.325, 17.673, 17.999, 18.329, 18.588, 18.885, 19.103, 19.345,
     E 19.537, 19.721, 19.884, 20.024, 20.145, 20.251, 20.330, 20.401,
     F 20.444, 20.477, 20.489, 20.483, 20.467, 20.427, 20.382, 20.310,
     G 20.236, 20.136, 20.036, 19.909, 19.785, 19.632, 19.486, 19.311,
     H 19.145, 18.949, 18.764, 18.551, 18.348, 18.119, 17.901, 17.659,
     I 17.428, 17.174, 16.931, 16.668, 16.415, 16.144, 15.883, 15.606,
     J 15.338, 15.058, 14.784, 14.501, 14.225, 13.941, 13.662, 13.378,
     K 13.098, 12.816, 12.536, 12.257, 11.978, 11.703, 11.427, 11.156,
     L 10.884, 10.618, 10.350, 10.090,  9.827,  9.574,  9.318,  9.072,
     M  8.822, 8.584, 8.340, 8.110, 7.874, 7.652, 7.424, 7.211, 6.990,
     N  6.785, 6.573, 6.377, 6.173, 5.986, 5.790, 5.612, 5.424, 5.255,
     O  5.075, 4.915, 4.744, 4.592, 4.429, 4.285, 4.130, 3.994, 3.847,
     P  3.719, 3.580, 3.459, 3.327, 3.214, 3.090, 2.983, 2.866, 2.766,
     Q  2.656, 2.562, 2.459, 2.372, 2.274, 2.193, 2.102, 2.025, 1.940,
     R  1.869, 1.790, 1.723, 1.649, 1.588, 1.518, 1.461, 1.397, 1.344,
     S  1.284, 1.235, 1.179, 1.134, 1.082, 1.040, 0.992, 0.954, 0.909 /
      DATA  ( CLDMOM(K), K = 160, 299 ) /
     T  0.873, 0.832, 0.799, 0.762, 0.731, 0.696, 0.668, 0.636, 0.610,
     U  0.581, 0.557, 0.530, 0.508, 0.483, 0.463, 0.440, 0.422, 0.401,
     V  0.384, 0.364, 0.349, 0.331, 0.317, 0.301, 0.288, 0.273, 0.262,
     W  0.248, 0.238, 0.225, 0.215, 0.204, 0.195, 0.185, 0.177, 0.167,
     X  0.160, 0.151, 0.145, 0.137, 0.131, 0.124, 0.118, 0.112, 0.107,
     Y  0.101, 0.097, 0.091, 0.087, 0.082, 0.079, 0.074, 0.071, 0.067,
     Z  0.064, 0.060, 0.057, 0.054, 0.052, 0.049, 0.047, 0.044, 0.042,
     A  0.039, 0.038, 0.035, 0.034, 0.032, 0.030, 0.029, 0.027, 0.026,
     B  0.024, 0.023, 0.022, 0.021, 0.020, 0.018, 0.018, 0.017, 0.016,
     C  0.015, 0.014, 0.013, 0.013, 0.012, 0.011, 0.011, 0.010, 0.009,
     D  0.009, 3*0.008, 2*0.007, 3*0.006, 4*0.005, 4*0.004, 6*0.003,
     E  9*0.002, 18*0.001 /
     
      DATA  ( AEROSOLMOM(K) , K = 1, 931 ) /
     A  7.927502e-01 , 7.051039e-01 , 5.585669e-01 , 5.010504e-01 ,
     B  4.384329e-01 , 4.055633e-01 , 3.805318e-01 , 3.565883e-01 ,
     C  3.406228e-01 , 3.254770e-01 , 3.095412e-01 , 2.994825e-01 , 
     D  2.847025e-01 , 2.758280e-01 , 2.636483e-01 , 2.543325e-01 , 
     E  2.448594e-01 , 2.353502e-01 , 2.273374e-01 , 2.188207e-01 ,
     F  2.110734e-01 , 2.039806e-01 , 1.964115e-01 , 1.902922e-01 ,
     F  1.833134e-01 , 1.776282e-01 , 1.715069e-01 , 1.660289e-01 ,
     F  1.606649e-01 , 1.555283e-01 , 1.506068e-01 , 1.459867e-01 ,
     F  1.413382e-01 , 1.371971e-01 , 1.328607e-01 , 1.290425e-01 ,
     F  1.250982e-01 , 1.214851e-01 , 1.179349e-01 , 1.145190e-01 ,
     F  1.112763e-01 , 1.081111e-01 , 1.050812e-01 , 1.021935e-01 ,
     F  9.935422e-02 , 9.669185e-02 , 9.405109e-02 , 9.154891e-02 ,
     F  8.911660e-02 , 8.675509e-02 , 8.450155e-02 , 8.229621e-02 ,
     F  8.019373e-02 , 7.816615e-02 , 7.617774e-02 , 7.429307e-02 ,
     F  7.242008e-02 , 7.065618e-02 , 6.891064e-02 , 6.725112e-02 ,
     F  6.561913e-02 , 6.405233e-02 , 6.252693e-02 , 6.106229e-02 ,
     F  5.961893e-02 , 5.824616e-02 , 5.688287e-02 , 5.559890e-02 ,
     F  5.431756e-02 , 5.310762e-02 , 5.190145e-02 , 5.075186e-02 ,
     F  4.961592e-02 , 4.853466e-02 , 4.746241e-02 , 4.644476e-02 ,
     F  4.542709e-02 , 4.446715e-02 , 4.350571e-02 , 4.259803e-02 ,
     F  4.168948e-02 , 4.082512e-02 , 3.996817e-02 , 3.914718e-02 ,
     F  3.833549e-02 , 3.755916e-02 , 3.678731e-02 , 3.605310e-02 ,
     F  3.531820e-02 , 3.462293e-02 , 3.392578e-02 , 3.326604e-02 ,
     F  3.260621e-02 , 3.197853e-02 , 3.135051e-02 , 3.075103e-02 ,
     F  3.015359e-02 , 2.958393e-02 , 2.901443e-02 , 2.847425e-02 ,
     F  2.792982e-02 , 2.741410e-02 , 2.689562e-02 , 2.640488e-02 ,
     F  2.591038e-02 , 2.544206e-02 , 2.497011e-02 , 2.452349e-02 ,
     F  2.407370e-02 , 2.364740e-02 , 2.321736e-02 , 2.280961e-02 ,
     F  2.239920e-02 , 2.200971e-02 , 2.161763e-02 , 2.124474e-02 ,
     F  2.086950e-02 , 2.051279e-02 , 2.015310e-02 , 1.981228e-02 ,
     F  1.946797e-02 , 1.914223e-02 , 1.881274e-02 , 1.850032e-02 ,
     F  1.818469e-02 , 1.788464e-02 , 1.758222e-02 , 1.729481e-02 ,
     F  1.700452e-02 , 1.672902e-02 , 1.645033e-02 , 1.618658e-02 ,
     F  1.591873e-02 , 1.566596e-02 , 1.540942e-02 , 1.516639e-02 ,
     F  1.492008e-02 , 1.468597e-02 , 1.444962e-02 , 1.422452e-02 ,
     F  1.399709e-02 , 1.378062e-02 , 1.356214e-02 , 1.335443e-02 ,
     F  1.314410e-02 , 1.294437e-02 , 1.274177e-02 , 1.255000e-02 ,
     F  1.235482e-02 , 1.217033e-02 , 1.198237e-02 , 1.180455e-02 ,
     F  1.162366e-02 , 1.145263e-02 , 1.127818e-02 , 1.111303e-02 ,
     F  1.094535e-02 , 1.078641e-02 , 1.062502e-02 , 1.047181e-02 ,
     F  1.031562e-02 , 1.016756e-02 , 1.001690e-02 , 9.874344e-03 ,
     F  9.729127e-03 , 9.591774e-03 , 9.452009e-03 , 9.319256e-03 ,
     F  9.183834e-03 , 9.055517e-03 , 8.924792e-03 , 8.801469e-03 ,
     F  8.675096e-03 , 8.555682e-03 , 8.433525e-03 , 8.318336e-03 ,
     F  8.200525e-03 , 8.089236e-03 , 7.975208e-03 , 7.867554e-03 ,
     F  7.757261e-03 , 7.653303e-03 , 7.546745e-03 , 7.446135e-03 ,
     F  7.343240e-03 , 7.245710e-03 , 7.146153e-03 , 7.051712e-03 ,
     F  6.955371e-03 , 6.864092e-03 , 6.770823e-03 , 6.682461e-03 ,
     F  6.592073e-03 , 6.506460e-03 , 6.419053e-03 , 6.336385e-03 ,
     F  6.251639e-03 , 6.171536e-03 , 6.089422e-03 , 6.011894e-03 ,
     F  5.932180e-03 , 5.856940e-03 , 5.779867e-03 , 5.706942e-03 ,
     F  5.632348e-03 , 5.561642e-03 , 5.489265e-03 , 5.420638e-03 ,
     F  5.350487e-03 , 5.283990e-03 , 5.215930e-03 , 5.151370e-03 ,
     F  5.085419e-03 , 5.022752e-03 , 4.958810e-03 , 4.898048e-03 ,
     F  4.835867e-03 , 4.776821e-03 , 4.716498e-03 , 4.659390e-03 ,
     F  4.600847e-03 , 4.545361e-03 , 4.488477e-03 , 4.434515e-03 ,
     F  4.379249e-03 , 4.326843e-03 , 4.273216e-03 , 4.222404e-03 ,
     F  4.170229e-03 , 4.120858e-03 , 4.070208e-03 , 4.022214e-03 ,
     F  3.973072e-03 , 3.926376e-03 , 3.878630e-03 , 3.833198e-03 ,
     F  3.786747e-03 , 3.742599e-03 , 3.697380e-03 , 3.654558e-03 ,
     F  3.610574e-03 , 3.568952e-03 , 3.526197e-03 , 3.485633e-03 ,
     F  3.444107e-03 , 3.404668e-03 , 3.364253e-03 , 3.325859e-03 ,
     F  3.286464e-03 , 3.249172e-03 , 3.210849e-03 , 3.174620e-03 ,
     F  3.137305e-03 , 3.101989e-03 , 3.065647e-03 , 3.031229e-03 ,
     F  2.995938e-03 , 2.962440e-03 , 2.928072e-03 , 2.895432e-03 ,
     F  2.861956e-03 , 2.830178e-03 , 2.797553e-03 , 2.766630e-03 ,
     F  2.734909e-03 , 2.704820e-03 , 2.673910e-03 , 2.644555e-03 ,
     F  2.614392e-03 , 2.585772e-03 , 2.556379e-03 , 2.528581e-03 ,
     F  2.499949e-03 , 2.472811e-03 , 2.444881e-03 , 2.418427e-03 ,
     F  2.391211e-03 , 2.365423e-03 , 2.338895e-03 , 2.313768e-03 ,
     F  2.287906e-03 , 2.263374e-03 , 2.238166e-03 , 2.214242e-03 ,
     F  2.189714e-03 , 2.166388e-03 , 2.142425e-03 , 2.119653e-03 ,
     F  2.096292e-03 , 2.074116e-03 , 2.051302e-03 , 2.029665e-03 ,
     F  2.007410e-03 , 1.986293e-03 , 1.964564e-03 , 1.943936e-03 ,
     F  1.922754e-03 , 1.902653e-03 , 1.881991e-03 , 1.862380e-03 ,
     F  1.842192e-03 , 1.823051e-03 , 1.803336e-03 , 1.784627e-03 ,
     F  1.765388e-03 , 1.747120e-03 , 1.728345e-03 , 1.710503e-03 ,
     F  1.692200e-03 , 1.674777e-03 , 1.656882e-03 , 1.639862e-03 ,
     F  1.622348e-03 , 1.605733e-03 , 1.588659e-03 , 1.572461e-03 ,
     F  1.555776e-03 , 1.539945e-03 , 1.523655e-03 , 1.508180e-03 ,
     F  1.492251e-03 , 1.477129e-03 , 1.461559e-03 , 1.446774e-03 ,
     F  1.431585e-03 , 1.417123e-03 , 1.402303e-03 , 1.388172e-03 ,
     F  1.373679e-03 , 1.359849e-03 , 1.345675e-03 , 1.332172e-03 ,
     F  1.318316e-03 , 1.305146e-03 , 1.291596e-03 , 1.278714e-03 ,
     F  1.265462e-03 , 1.252857e-03 , 1.239909e-03 , 1.227579e-03 ,
     F  1.214919e-03 , 1.202858e-03 , 1.190470e-03 , 1.178686e-03 ,
     F  1.166572e-03 , 1.155049e-03 , 1.143193e-03 , 1.131914e-03 ,
     F  1.120320e-03 , 1.109289e-03 , 1.097957e-03 , 1.087167e-03 ,
     F  1.076091e-03 , 1.065525e-03 , 1.054691e-03 , 1.044347e-03 ,
     F  1.033743e-03 , 1.023627e-03 , 1.013238e-03 , 1.003330e-03 ,
     F  9.931699e-04 , 9.834933e-04 , 9.735461e-04 , 9.640734e-04 ,
     F  9.543410e-04 , 9.450750e-04 , 9.355414e-04 , 9.264655e-04 ,
     F  9.171349e-04 , 9.082570e-04 , 8.991263e-04 , 8.904461e-04 ,
     F  8.815043e-04 , 8.729967e-04 , 8.642440e-04 , 8.559101e-04 ,
     F  8.473352e-04 , 8.391797e-04 , 8.307864e-04 , 8.227958e-04 ,
     F  8.145851e-04 , 8.067586e-04 , 7.987195e-04 , 7.910583e-04 ,
     F  7.831822e-04 , 7.756758e-04 , 7.679565e-04 , 7.606141e-04 ,
     F  7.530494e-04 , 7.458543e-04 , 7.384427e-04 , 7.313900e-04 ,
     F  7.241381e-04 , 7.172347e-04 , 7.101381e-04 , 7.033708e-04 ,
     F  6.964118e-04 , 6.897753e-04 , 6.829472e-04 , 6.764508e-04 ,
     F  6.697634e-04 , 6.634054e-04 , 6.568558e-04 , 6.506254e-04 ,
     F  6.442058e-04 , 6.380909e-04 , 6.317958e-04 , 6.257972e-04 ,
     F  6.196380e-04 , 6.137709e-04 , 6.077350e-04 , 6.019845e-04 ,
     F  5.960632e-04 , 5.904211e-04 , 5.846107e-04 , 5.790788e-04 ,
     F  5.733900e-04 , 5.679753e-04 , 5.624048e-04 , 5.570949e-04 ,
     F  5.516313e-04 , 5.464169e-04 , 5.410606e-04 , 5.359464e-04 ,
     F  5.306946e-04 , 5.256821e-04 , 5.205367e-04 , 5.156235e-04 ,
     F  5.105777e-04 , 5.057623e-04 , 5.008071e-04 , 4.960873e-04 ,
     F  4.912318e-04 , 4.866032e-04 , 4.818416e-04 , 4.773064e-04 ,
     F  4.726325e-04 , 4.681846e-04 , 4.636030e-04 , 4.592416e-04 ,
     F  4.547502e-04 , 4.504730e-04 , 4.460664e-04 , 4.418695e-04 ,
     F  4.375444e-04 , 4.334282e-04 , 4.291862e-04 , 4.251489e-04 ,
     F  4.209907e-04 , 4.170305e-04 , 4.129539e-04 , 4.090678e-04 ,
     F  4.050685e-04 , 4.012541e-04 , 3.973321e-04 , 3.935865e-04 ,
     F  3.897367e-04 , 3.860662e-04 , 3.822920e-04 , 3.786927e-04 ,
     F  3.749875e-04 , 3.714541e-04 , 3.678193e-04 , 3.643549e-04 ,
     F  3.607890e-04 , 3.573884e-04 , 3.538888e-04 , 3.505509e-04 ,
     F  3.471174e-04 , 3.438431e-04 , 3.404755e-04 , 3.372617e-04 ,
     F  3.339571e-04 , 3.308039e-04 , 3.275575e-04 , 3.244596e-04 ,
     F  3.212748e-04 , 3.182396e-04 , 3.151143e-04 , 3.121337e-04 ,
     F  3.090664e-04 , 3.061386e-04 , 3.031311e-04 , 3.002628e-04 ,
     F  2.973095e-04 , 2.944935e-04 , 2.915921e-04 , 2.888275e-04 ,
     F  2.859786e-04 , 2.832651e-04 , 2.804713e-04 , 2.778060e-04 ,
     F  2.750671e-04 , 2.724512e-04 , 2.697626e-04 , 2.671940e-04 ,
     F  2.645519e-04 , 2.620313e-04 , 2.594345e-04 , 2.569612e-04 ,
     F  2.544127e-04 , 2.519885e-04 , 2.494896e-04 , 2.471090e-04 ,
     F  2.446548e-04 , 2.423144e-04 , 2.399059e-04 , 2.376079e-04 ,
     F  2.352415e-04 , 2.329854e-04 , 2.306624e-04 , 2.284511e-04 ,
     F  2.261706e-04 , 2.239998e-04 , 2.217581e-04 , 2.196270e-04 ,
     F  2.174308e-04 , 2.153401e-04 , 2.131812e-04 , 2.111244e-04 ,
     F  2.090049e-04 , 2.069849e-04 , 2.049068e-04 , 2.029271e-04 ,
     F  2.008845e-04 , 1.989382e-04 , 1.969312e-04 , 1.950223e-04 ,
     F  1.930533e-04 , 1.911817e-04 , 1.892469e-04 , 1.874081e-04 ,
     F  1.855078e-04 , 1.837013e-04 , 1.818365e-04 , 1.800644e-04 ,
     F  1.782353e-04 , 1.764961e-04 , 1.746982e-04 , 1.729894e-04 ,
     F  1.712238e-04 , 1.695476e-04 , 1.678130e-04 , 1.661661e-04 ,
     F  1.644641e-04 , 1.628483e-04 , 1.611774e-04 , 1.595893e-04 ,
     F  1.579503e-04 , 1.563900e-04 , 1.547784e-04 , 1.532462e-04 ,
     F  1.516649e-04 , 1.501618e-04 , 1.486082e-04 , 1.471312e-04 ,
     F  1.456040e-04 , 1.441531e-04 , 1.426526e-04 , 1.412287e-04 ,
     F  1.397560e-04 , 1.383584e-04 , 1.369118e-04 , 1.355399e-04 ,
     F  1.341181e-04 , 1.327698e-04 , 1.313746e-04 , 1.300523e-04 ,
     F  1.286819e-04 , 1.273839e-04 , 1.260365e-04 , 1.247606e-04 ,
     F  1.234376e-04 , 1.221859e-04 , 1.208875e-04 , 1.196570e-04 ,
     F  1.183839e-04 , 1.171762e-04 , 1.159280e-04 , 1.147432e-04 ,
     F  1.135154e-04 , 1.123510e-04 , 1.111435e-04 , 1.100013e-04 ,
     F  1.088175e-04 , 1.076984e-04 , 1.065383e-04 , 1.054386e-04 ,
     F  1.042985e-04 , 1.032174e-04 , 1.020982e-04 , 1.010379e-04 ,
     F  9.993908e-05 , 9.889788e-05 , 9.781914e-05 , 9.679769e-05 ,
     F  9.573671e-05 , 9.473292e-05 , 9.369051e-05 , 9.270528e-05 ,
     F  9.168037e-05 , 9.071226e-05 , 8.970843e-05 , 8.875863e-05 ,
     F  8.777218e-05 , 8.683788e-05 , 8.586872e-05 , 8.495190e-05 ,
     F  8.400022e-05 , 8.310165e-05 , 8.216716e-05 , 8.128495e-05 ,
     F  8.036833e-05 , 7.950313e-05 , 7.860333e-05 , 7.775375e-05 ,
     F  7.687158e-05 , 7.603877e-05 , 7.517350e-05 , 7.435624e-05 ,
     F  7.350729e-05 , 7.270478e-05 , 7.187072e-05 , 7.108370e-05 ,
     F  7.026604e-05 , 6.949349e-05 , 6.869102e-05 , 6.793244e-05 ,
     F  6.714419e-05 , 6.639834e-05 , 6.562381e-05 , 6.489106e-05 ,
     F  6.413145e-05 , 6.341312e-05 , 6.266680e-05 , 6.196132e-05 ,
     F  6.122828e-05 , 6.053621e-05 , 5.981623e-05 , 5.913771e-05 ,
     F  5.843147e-05 , 5.776637e-05 , 5.707421e-05 , 5.642160e-05 ,
     F  5.574340e-05 , 5.510413e-05 , 5.443922e-05 , 5.381224e-05 ,
     F  5.315975e-05 , 5.254624e-05 , 5.190514e-05 , 5.130341e-05 ,
     F  5.067460e-05 , 5.008460e-05 , 4.946771e-05 , 4.888769e-05 ,
     F  4.828216e-05 , 4.771216e-05 , 4.711765e-05 , 4.655812e-05 ,
     F  4.597436e-05 , 4.542572e-05 , 4.485311e-05 , 4.431612e-05 ,
     F  4.375467e-05 , 4.322766e-05 , 4.267768e-05 , 4.216143e-05 ,
     F  4.162293e-05 , 4.111710e-05 , 4.058987e-05 , 4.009505e-05 ,
     F  3.957835e-05 , 3.909394e-05 , 3.858707e-05 , 3.811139e-05 ,
     F  3.761352e-05 , 3.714756e-05 , 3.665868e-05 , 3.620095e-05 ,
     F  3.572043e-05 , 3.527204e-05 , 3.480099e-05 , 3.436220e-05 ,
     F  3.389977e-05 , 3.346856e-05 , 3.301625e-05 , 3.259501e-05 ,
     F  3.215319e-05 , 3.174109e-05 , 3.130930e-05 , 3.090622e-05 ,
     F  3.048272e-05 , 3.008785e-05 , 2.967240e-05 , 2.928530e-05 ,
     F  2.887819e-05 , 2.849917e-05 , 2.809978e-05 , 2.772740e-05 ,
     F  2.733531e-05 , 2.696935e-05 , 2.658492e-05 , 2.622725e-05 ,
     F  2.585066e-05 , 2.550086e-05 , 2.513285e-05 , 2.479231e-05 ,
     F  2.443286e-05 , 2.410064e-05 , 2.374950e-05 , 2.342429e-05 ,
     F  2.308040e-05 , 2.276168e-05 , 2.242360e-05 , 2.211059e-05 ,
     F  2.177893e-05 , 2.147249e-05 , 2.114716e-05 , 2.084720e-05 ,
     F  2.052859e-05 , 2.023509e-05 , 1.992404e-05 , 1.963780e-05 ,
     F  1.933482e-05 , 1.905600e-05 , 1.876049e-05 , 1.848875e-05 ,
     F  1.819971e-05 , 1.793401e-05 , 1.764972e-05 , 1.738882e-05 ,
     F  1.710970e-05 , 1.685361e-05 , 1.658017e-05 , 1.632987e-05 ,
     F  1.606314e-05 , 1.581913e-05 , 1.555909e-05 , 1.532145e-05 ,
     F  1.506815e-05 , 1.483749e-05 , 1.459039e-05 , 1.436530e-05 ,
     F  1.412330e-05 , 1.390295e-05 , 1.366516e-05 , 1.344888e-05 ,
     F  1.321591e-05 , 1.300450e-05 , 1.277688e-05 , 1.257097e-05 ,
     F  1.234936e-05 , 1.214948e-05 , 1.193433e-05 , 1.174025e-05 ,
     F  1.153066e-05 , 1.134135e-05 , 1.113653e-05 , 1.095086e-05 ,
     F  1.074985e-05 , 1.056829e-05 , 1.037114e-05 , 1.019388e-05 ,
     F  1.000163e-05 , 9.829493e-06 , 9.643095e-06 , 9.476699e-06 ,
     F  9.296248e-06 , 9.134731e-06 , 8.958884e-06 , 8.800983e-06 ,
     F  8.628064e-06 , 8.472977e-06 , 8.303275e-06 , 8.152198e-06 ,
     F  7.987009e-06 , 7.840709e-06 , 7.680917e-06 , 7.539927e-06 ,
     F  7.385317e-06 , 7.248788e-06 , 7.098165e-06 , 6.964900e-06 ,
     F  6.817064e-06 , 6.686208e-06 , 6.541129e-06 , 6.413637e-06 ,
     F  6.273135e-06 , 6.150548e-06 , 6.015199e-06 , 5.897371e-06 ,
     F  5.766461e-06 , 5.652240e-06 , 5.524264e-06 , 5.412079e-06 ,
     F  5.286346e-06 , 5.176319e-06 , 5.054223e-06 , 4.948351e-06 ,
     F  4.831190e-06 , 4.730043e-06 , 4.617059e-06 , 4.519427e-06 ,
     F  4.409092e-06 , 4.313564e-06 , 4.205232e-06 , 4.111818e-06 ,
     F  4.006410e-06 , 3.916665e-06 , 3.816072e-06 , 3.730697e-06 ,
     F  3.634243e-06 , 3.552065e-06 , 3.457489e-06 , 3.376344e-06 ,
     F  3.283028e-06 , 3.204318e-06 , 3.115015e-06 , 3.041145e-06 ,
     F  2.956268e-06 , 2.885322e-06 , 2.802949e-06 , 2.733604e-06 ,
     F  2.652685e-06 , 2.584616e-06 , 2.506095e-06 , 2.441808e-06 ,
     F  2.368118e-06 , 2.308084e-06 , 2.236837e-06 , 2.177202e-06 ,
     F  2.106341e-06 , 2.048448e-06 , 1.981465e-06 , 1.927859e-06 ,
     F  1.864382e-06 , 1.812484e-06 , 1.750170e-06 , 1.699771e-06 ,
     F  1.639766e-06 , 1.592227e-06 , 1.535268e-06 , 1.490237e-06 ,
     F  1.435341e-06 , 1.391960e-06 , 1.339043e-06 , 1.297546e-06 ,
     F  1.247063e-06 , 1.207794e-06 , 1.159606e-06 , 1.122096e-06 ,
     F  1.075953e-06 , 1.040287e-06 , 9.960538e-07 , 9.624508e-07 ,
     F  9.202548e-07 , 8.886740e-07 , 8.483949e-07 , 8.187408e-07 ,
     F  7.802536e-07 , 7.521428e-07 , 7.155470e-07 , 6.892034e-07 ,
     F  6.545744e-07 , 6.298993e-07 , 5.969407e-07 , 5.736458e-07 ,
     F  5.425084e-07 , 5.209553e-07 , 4.915092e-07 , 4.714161e-07 ,
     F  4.436078e-07 , 4.249280e-07 , 3.987779e-07 , 3.815864e-07 ,
     F  3.569974e-07 , 3.411631e-07 , 3.180942e-07 , 3.034575e-07 ,
     F  2.818743e-07 , 2.684621e-07 , 2.484024e-07 , 2.363313e-07 ,
     F  2.176626e-07 , 2.066189e-07 , 1.893025e-07 , 1.794621e-07 ,
     F  1.633714e-07 , 1.546060e-07 , 1.397982e-07 , 1.321178e-07 ,
     F  1.185694e-07 , 1.119363e-07 , 9.946173e-08 , 9.365221e-08 ,
     F  8.235258e-08 , 7.754409e-08 , 6.734401e-08 , 6.328659e-08 ,
     F  5.406462e-08 , 5.076214e-08 , 4.256807e-08 , 4.007295e-08 ,
     F  3.283177e-08 , 3.093717e-08 , 2.456445e-08 , 2.331709e-08 ,
     F  1.780965e-08 , 1.716962e-08 , 1.244885e-08 , 1.229666e-08 ,
     F  8.324811e-09 , 8.605210e-09 , 5.325729e-09 , 5.826041e-09 ,
     F  3.248923e-09 , 3.891680e-09 , 1.913883e-09 , 2.506743e-09 ,
     F  1.048926e-09 , 1.620911e-09 , 5.937370e-10 , 9.472280e-10 ,
     F  2.921713e-10 , 5.788684e-10 , 1.613183e-10 , 3.366239e-10 ,
     F  7.347505e-11 , 1.789125e-10 , 3.728341e-11 /


      IF ( IPHAS.LT.1 .OR. IPHAS.GT.6 )
     &     CALL ERRMSG( 'GETMOM--bad input variable IPHAS',.TRUE.)

      IF ( IPHAS.EQ.3 .AND. (GG.LE.-1.0 .OR. GG.GE.1.0) )
     &     CALL ERRMSG( 'GETMOM--bad input variable GG',.TRUE.)

      IF ( NMOM.LT.2 )
     &     CALL ERRMSG( 'GETMOM--bad input variable NMOM',.TRUE.)


      PMOM(0) = 1.0
      DO  10  K = 1, NMOM
         PMOM(K) = 0.0
   10 CONTINUE


      IF ( IPHAS.EQ.2 )  THEN
c                                       ** Rayleigh phase function
         PMOM(2) = 0.1

      ELSE IF ( IPHAS.EQ.3 ) THEN
c                                       ** Henyey-Greenstein phase fcn
         DO  20  K = 1, NMOM
            PMOM(K) = GG**K
   20    CONTINUE

      ELSE IF ( IPHAS.EQ.4 ) THEN
c                                        ** Haze-L phase function
         DO  30  K = 1, MIN(82,NMOM)
            PMOM(K) = HAZELM(K) / ( 2*K+1 )
   30    CONTINUE

      ELSE IF ( IPHAS.EQ.5 ) THEN
c                                        ** Cloud C.1 phase function
         DO  40  K = 1, MIN(298,NMOM)
            PMOM(K) = CLDMOM(K) / ( 2*K+1 )
40       CONTINUE

      ELSE IF ( IPHAS.EQ.6 ) THEN
c                                        ** Aerosol phase function          
         DO  50  K = 1, MIN(931,NMOM)
            PMOM(K) = AEROSOLMOM(K) 
50       CONTINUE

      END IF

      END


      SUBROUTINE  FLUX_ANALYTIC(BRDF_TYPE, BRDF_ARG, 
     &                        UMU0, FBEAM, NPHI, NUMU, SSALB,
     &                        FLUX_UP, DFDT )
        

        REAL       GWTPHI(NPHI), GPHI(NPHI) 
        REAL       GWTMU(NUMU), GMU(NUMU)
        INTEGER    BRDF_TYPE
        REAL       BRDF_ARG(4)
        REAL       FLUX_UP
        REAL       WVNML0
        REAL       WVNMHI
        REAL       PI
        REAL       SSALB
        REAL       I_AVER_UP, I_AVER_DN, DFDT

        WVNML0 = 0.0
        WVNMHI = 0.0
        PI     = 2.*ASIN(1.)

        CALL  QGAUSN2( NPHI/2, GPHI, GWTPHI )
        CALL  QGAUSN2( NUMU, GMU, GWTMU )

        GPHI   = GPHI*PI
        GWTPHI = GWTPHI*PI

        GMU    = GMU
        GWTMU  = GWTMU


        FLUX_UP = 0.0
        I_AVER_UP = 0.0
        DO I = 1, NPHI/2
          DO J = 1, NUMU
            FLUX_UP = FLUX_UP +
     &                UMU0 * BDREF( GMU(J),  
     &                UMU0, GPHI(I), BRDF_TYPE, BRDF_ARG )     
     &                * GMU(J) * GWTMU(J) * GWTPHI(I)
     
            I_AVER_UP = I_AVER_UP +
     &                UMU0 * BDREF( GMU(J),  
     &                UMU0, GPHI(I), BRDF_TYPE, BRDF_ARG )     
     &                * GWTMU(J) * GWTPHI(I)

          ENDDO
        ENDDO
        FLUX_UP = 2. * FLUX_UP * FBEAM
        
        I_AVER_UP = 2. * I_AVER_UP / (2.*PI)
        I_AVER_DN = FBEAM / (2.*PI)
        
        DFDT      = 4.*PI*(1.-SSALB)*( 0.5*(I_AVER_UP + I_AVER_DN) )

      END


      SUBROUTINE  PRTFIN( UTAU, NTAU, UMU, NUMU, PHI, NPHI, MAXULV,
     &                    MAXUMU, ONLYFL, RFLDIR, RFLDN, FLUP, DFDT,
     &                    UU, TSTFIR, TSTFDN, TSTFUP, TSTDFD, TSTUU,
     &                    MXTAU, MXMU, MXPHI, NTEST, NPASS )

c        Print DISORT results and, directly beneath them, their
c        ratios to the correct answers;  print number of non-unit
c        ratios that occur but try to count just the cases where
c        there is a real disagreement and not those where flux or
c        intensity are down at their noise level (defined as 10^(-6)
c        times their maximum value).  d(flux)/d(tau) is treated the
c        same as fluxes in this noise estimation even though it
c        is a different type of quantity (although with flux units).

c     INPUT :   TSTFIR  correct direct flux
c               TSTFDN  correct diffuse down flux
c               TSTFUP  correct diffuse up flux
c               TSTDFD  correct d(flux)/d(optical depth)
c               TSTUU   correct intensity
c               (remaining input = DISORT I/O variables)

c --------------------------------------------------------------------

c     .. Parameters ..

      INTEGER   MAXRAT
      PARAMETER ( MAXRAT = 100 )
c     ..
c     .. Scalar Arguments ..

      LOGICAL   ONLYFL
      INTEGER   MXPHI, MXMU, MXTAU, MAXULV, MAXUMU, NPHI, NTAU, NUMU
c     ..
c     .. Array Arguments ..

      REAL      DFDT( * ), FLUP( * ), PHI( * ), RFLDIR( * ), RFLDN( * ),
     &          TSTDFD( * ), TSTFDN( * ), TSTFIR( * ), TSTFUP( * ),
     &          TSTUU( MXTAU, MXMU, MXPHI ), UMU( * ), UTAU( * ),
     &          UU( MAXUMU, MAXULV, * )
c     ..
c     .. Local Scalars ..

      INTEGER  IU, J, LU, NUMBAD
      REAL     FLXMAX, FNOISE, RAT, RAT1, RAT2, RAT3, RAT4, UMAX, UNOISE
c     ..
c     .. Local Arrays ..

      REAL      RATV( MAXRAT )
c     ..
c     .. External Functions ..

      REAL      RATIO
      EXTERNAL  RATIO
c     ..
c     .. External Subroutines ..

      EXTERNAL  ERRMSG
c     ..
c     .. Statement Functions ..

      LOGICAL   BADRAT
c     ..
c     .. Statement Function definitions ..

      BADRAT( RAT ) = (RAT.LT.0.99) .OR. (RAT.GT.1.01)
c     ..

      IF ( NTAU.GT.MXTAU .OR. NUMU.GT.MXMU .OR. NPHI.GT.MXPHI )  CALL
     &   ERRMSG( 'PRTFIN--out of bounds in comparator arrays', .TRUE.)

      FLXMAX = 0.0
      DO 5  LU = 1, NTAU
         FLXMAX = MAX( FLXMAX, TSTFIR(LU), TSTFDN(LU), TSTFUP(LU) )
 5    CONTINUE
      FNOISE = 1.E-6 * FLXMAX
      IF( FLXMAX.LE.0.0 )
     &    CALL ERRMSG( 'PRTFIN--all fluxes zero or negative', .FALSE.)
      IF( FNOISE.LE.0.0 )
     &    CALL ERRMSG( 'PRTFIN--all fluxes near underflowing', .FALSE.)

      NUMBAD = 0

      WRITE(*,'(//,A,/,A,/,A)')
     &  '                  <-------------- FLUXES -------------->',
     &  '    Optical       Downward       Downward         Upward'//
     &  '    d(Net Flux)',
     &  '      Depth         Direct        Diffuse        Diffuse'//
     &  '    / d(Op Dep)'

      DO 10  LU = 1, NTAU

c         WRITE( *,'(0P,F11.4,1P,4E15.4)')  UTAU(LU), RFLDIR(LU),
         WRITE( *,'(0P,F11.5,1P,4E15.5)')  UTAU(LU), RFLDIR(LU),
     &          RFLDN(LU), FLUP(LU), DFDT(LU)
         RAT1 = RATIO( RFLDIR(LU), TSTFIR(LU) )
         RAT2 = RATIO( RFLDN(LU),  TSTFDN(LU) )
         RAT3 = RATIO(  FLUP(LU),  TSTFUP(LU) )
         RAT4 = RATIO(  DFDT(LU),  TSTDFD(LU) )
c         WRITE( *,'(11X,4( ''    ('',F9.4,'')''))')
c         WRITE( *,'(11X,4( ''   ('',G11.5,'')''))')
         WRITE( *,'(11X,4(''  ('',F11.5,'')''))')
     &          RAT1, RAT2, RAT3, RAT4

         IF( BADRAT(RAT1) .AND. ABS(RFLDIR(LU)).GT.FNOISE 
     &      .AND. TSTFIR(LU) .NE. 0. )
     &        NUMBAD = NUMBAD+1
         IF( BADRAT(RAT2) .AND. ABS(RFLDN(LU)).GT.FNOISE 
     &      .AND. TSTFDN(LU) .NE. 0. )
     &        NUMBAD = NUMBAD+1
         IF( BADRAT(RAT3) .AND. ABS(FLUP(LU)).GT.FNOISE 
     &      .AND. TSTFUP(LU) .NE. 0. )
     &        NUMBAD = NUMBAD+1
         IF( BADRAT(RAT4) .AND. ABS(DFDT(LU)).GT.FNOISE 
     &      .AND. TSTDFD(LU) .NE. 0. )
     &        NUMBAD = NUMBAD+1


10    CONTINUE


      IF ( ONLYFL )  GO TO 100

      IF ( NUMU.GT.MAXRAT .OR. NPHI.GT.MAXRAT )
     &     CALL ERRMSG( 'PRTFIN--increase parameter MAXRAT', .TRUE.)


c                                       ** Print intensities

      IF ( NPHI.GT.8 ) CALL ERRMSG
     &      ( 'PRTFIN--intensity FORMATs inadequate',.FALSE.)

      UMAX = 0.0
      DO 36  LU = 1, NTAU
         DO 35  IU = 1, NUMU
            DO 34  J = 1, NPHI
               UMAX = MAX( UMAX, TSTUU(LU,IU,J) )
 34         CONTINUE
 35      CONTINUE
 36   CONTINUE
      UNOISE = 1.E-6 * UMAX
      IF( UMAX.LE.0.0 )  CALL ERRMSG
     &     ( 'PRTFIN--all intensities zero or negative',.FALSE.)
      IF( UNOISE.LE.0.0 ) CALL ERRMSG
     &     ( 'PRTFIN--all intensities near underflowing',.FALSE.)

      WRITE( *,'(//,A,//,A,/,A,/,A,8(F10.1,4X))' )
     &            ' ********  I N T E N S I T I E S  *********',
     &            '             Polar   Azimuthal Angles (Degrees)',
     &            '   Optical   Angle',
     &            '     Depth  Cosine', ( PHI(J), J = 1, NPHI )

      DO 60  LU = 1, NTAU

         DO 50  IU = 1, NUMU

c            IF( IU.EQ.1 ) WRITE( *,'(/,0P,F10.3,F8.3,1P,8E14.4)')
            IF( IU.EQ.1 ) WRITE( *,'(/,0P,F10.3,F8.3,1P,8E16.5)')
     &          UTAU(LU), UMU(IU), ( UU( IU,LU,J ), J = 1, NPHI )

c            IF( IU.GT.1 ) WRITE( *,'(10X,0P,F8.3, 1P,8E14.4)')
            IF( IU.GT.1 ) WRITE( *,'(10X,0P,F8.3, 1P,8E16.5)')
     &                    UMU(IU), ( UU( IU,LU,J ), J = 1, NPHI )

            DO 40  J = 1, NPHI
               RATV(J) = RATIO( UU(IU,LU,J), TSTUU(LU,IU,J) )
               IF( BADRAT(RATV(J)) .AND. ABS(UU(IU,LU,J)).GT.UNOISE )
     &                  NUMBAD = NUMBAD + 1
   40       CONTINUE

c            WRITE( *,'(18X, 8(:,''   ('',F9.4,'')''))')
            WRITE( *,'(18X, 8(:,''   ('',F11.5,'')''))')
     &           ( RATV(J), J = 1, NPHI )

   50    CONTINUE
   60 CONTINUE

  100 CONTINUE
      IF( NUMBAD.GT.0 ) THEN
          WRITE( *,300)  ' ====  ', NUMBAD,
     &    '  SERIOUSLY NON-UNIT RATIOS    ===='
      ELSE
          NPASS = NPASS+1
      ENDIF
      NTEST = NTEST+1
      RETURN

300   FORMAT( //,1X,45('='),/,A,I4,A,/,1X,45('=') )
      END


      SUBROUTINE PRTFIN2(NTAU,NUMU,MAXUMU,MAXULV,
     &                         UTAU, UMU, UU, NPHI,PHI)

      INTEGER  NTAU, NUMU, NPHI
      INTEGER  LU, IU
      REAL     UTAU(*), UMU(*), UU(MAXUMU, MAXULV, *)
      REAL     PHI(*)

      OPEN(UNIT=10,FILE='./INTENSITY.DAT')
      DO LU = 1, NTAU
        DO IU = 1, NUMU
C         WRITE('(/,0P,F10.3,1P,8E14.4)')
          WRITE(10,*) 
     &      UTAU(LU),UMU(IU),(UU(IU,LU,J),J=1,NPHI)
        ENDDO
      ENDDO
      WRITE(10, '(5(F10.3,4X))')
     &   (PHI(J), J = 1,NPHI)

      END


      BLOCK DATA  CHEKDO

c       Correct answers to test problems ( as produced by DISORT
c       running entirely in double precision (14 significant digits)
c       on a Digital Alpha Workstation computer ).

c     .. Parameters ..

      INTEGER   MXPROB, MXCASE, MXTAU, MXMU, MXPHI
      PARAMETER ( MXPROB = 15, MXCASE = 8, MXTAU = 5, MXMU = 15,
     &            MXPHI = 3 )
c     ..
c     .. Local Scalars ..

      INTEGER   I, J, K
c     ..
c     .. Common blocks ..

      COMMON    / DOCHEK / TSTFIR, TSTFDN, TSTFUP, TSTDFD, TSTUU

      REAL      TSTDFD( MXTAU, MXCASE, MXPROB ),
     &          TSTFDN( MXTAU, MXCASE, MXPROB ),
     &          TSTFIR( MXTAU, MXCASE, MXPROB ),
     &          TSTFUP( MXTAU, MXCASE, MXPROB ),
     &          TSTUU( MXTAU, MXMU, MXPHI, MXCASE, MXPROB )
c     ..

c ********************* Test Case 1A *********************************

      DATA (TSTFIR(I,1,1), I = 1, 2) / 3.14159E+00, 2.29844E+00 /
      DATA (TSTFDN(I,1,1), I = 1, 2) / 0.0, 7.94108E-02 /
      DATA (TSTFUP(I,1,1), I = 1, 2) / 7.99451E-02, 0.0 /
      DATA (TSTDFD(I,1,1), I = 1, 2) / 2.54067E+01, 1.86531E+01 /
      DATA ((TSTUU(I,J,1,1,1), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 1.17771E-01, 2.64170E-02, 1.34041E-02,
     &    1.33826E-02, 2.63324E-02, 1.15898E-01, 3*0.0 /

c ********************* Test Case 1B *********************************

      DATA (TSTFIR(I,2,1), I = 1, 2) / 3.14159E+00, 2.29844E+00 /
      DATA (TSTFDN(I,2,1), I = 1, 2) / 0.0, 4.20233E-01 /
      DATA (TSTFUP(I,2,1), I = 1, 2) / 4.22922E-01, 0.0 /
      DATA (TSTDFD(I,2,1), I = 1, 2) / 2*0.0 /
      DATA ((TSTUU(I,J,1,2,1), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 6.22884E-01, 1.39763E-01, 7.09192E-02,
     &    7.08109E-02, 1.39337E-01, 6.13458E-01, 3*0.0 /

c ********************* Test Case 1C *********************************

      DATA (TSTFIR(I,3,1), I = 1, 2) / 2*0.0 /
      DATA (TSTFDN(I,3,1), I = 1, 2) / 3.14159E+00, 3.04897E+00 /
      DATA (TSTFUP(I,3,1), I = 1, 2) / 9.06556E-02, 0.0 /
      DATA (TSTDFD(I,3,1), I = 1, 2) / 6.66870E-02, 5.88936E-02 /
      DATA ((TSTUU(I,J,1,3,1), J = 1, 6), I = 1, 2)
     &  / 3*1.0, 1.33177E-01, 2.99879E-02, 1.52233E-02,
     &    9.84447E-01, 9.69363E-01, 8.63946E-01, 3*0.0 /

c ********************* Test Case 1D *********************************

      DATA (TSTFIR(I,4,1), I = 1, 2) / 3.14159E+00, 0.00000E+00 /
      DATA (TSTFDN(I,4,1), I = 1, 2) / 2*0.0 /
      DATA (TSTFUP(I,4,1), I = 1, 2) / 2.59686E-01, 0.0 /
      DATA (TSTDFD(I,4,1), I = 1, 2) / 2.57766E+01, 0.0 /
      DATA ((TSTUU(I,J,1,4,1), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 2.62972E-01, 9.06967E-02, 5.02853E-02,
     &    1.22980E-15, 1.30698E-17, 6.88840E-18, 3*0.0 /

c ********************* Test Case 1E *********************************

      DATA (TSTFIR(I,5,1), I = 1, 2) / 3.14159E+00, 0.00000E+00 /
      DATA (TSTFDN(I,5,1), I = 1, 2) / 0.0, 6.76954E-02 /
      DATA (TSTFUP(I,5,1), I = 1, 2) / 3.07390E+00, 0.0 /
      DATA (TSTDFD(I,5,1), I = 1, 2) / 2*0.0 /
      DATA ((TSTUU(I,J,1,5,1), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 1.93321E+00, 1.02732E+00, 7.97199E-01,
     &    2.71316E-02, 1.87805E-02, 1.16385E-02, 3*0.0 /

c ********************* Test Case 1F *********************************

      DATA (TSTFIR(I,6,1), I = 1, 2) / 2*0.0 /
      DATA (TSTFDN(I,6,1), I = 1, 2) / 3.14159E+00, 4.60048E-03 /
      DATA (TSTFUP(I,6,1), I = 1, 2) / 2.49618E+00, 0.0 /
      DATA (TSTDFD(I,6,1), I = 1, 2) / 1.14239E-01, 7.93633E-05 /
      DATA ((TSTUU(I,J,1,6,1), J = 1, 6), I = 1, 2)
     &  / 3*1.0, 8.77510E-01, 8.15136E-01, 7.52715E-01,
     &    1.86840E-03, 1.26492E-03, 7.79280E-04, 3*0.0 /


c ********************* Test Case 2A *********************************

      DATA (TSTFIR(I,1,2), I = 1, 2) / 2.52716E-01, 2.10311E-02 /
      DATA (TSTFDN(I,1,2), I = 1, 2) / 0.0, 4.41791E-02 /
      DATA (TSTFUP(I,1,2), I = 1, 2) / 5.35063E-02, 0.0 /
      DATA (TSTDFD(I,1,2), I = 1, 2) / 1.66570E+00, 1.89848E-01 /
      DATA ((TSTUU(I,J,1,1,2), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 1.61796E-01, 2.11501E-02, 7.86713E-03,
     &    7.71897E-03, 2.00778E-02, 2.57685E-02, 3*0.0 /

c ********************* Test Case 2B *********************************

      DATA (TSTFIR(I,2,2), I = 1, 2) / 2.52716E-01, 2.10311E-02 /
      DATA (TSTFDN(I,2,2), I = 1, 2) / 0.0, 1.06123E-01 /
      DATA (TSTFUP(I,2,2), I = 1, 2) / 1.25561E-01, 0.0 /
      DATA (TSTDFD(I,2,2), I = 1, 2) / 2*0.0 /
      DATA ((TSTUU(I,J,1,2,2), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 3.47678E-01, 4.87120E-02, 1.89387E-02,
     &    1.86027E-02, 4.64061E-02, 6.77603E-02, 3*0.0 /

c ********************* Test Case 2C *********************************

      DATA (TSTFIR(I,3,2), I = 1, 2) / 2.52716E-01, 2.56077E-28 /
      DATA (TSTFDN(I,3,2), I = 1, 2) / 0.0, 2.51683E-04 /
      DATA (TSTFUP(I,3,2), I = 1, 2) / 6.24730E-02, 0.0 /
      DATA (TSTDFD(I,3,2), I = 1, 2) / 1.67462E+00, 1.75464E-04 /
      DATA ((TSTUU(I,J,1,3,2), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 1.62566E-01, 2.45786E-02, 1.01498E-02,
     &    1.70004E-04, 3.97168E-05, 1.32472E-05, 3*0.0 /

c ********************* Test Case 2D *********************************

      DATA (TSTFIR(I,4,2), I = 1, 2) / 2.52716E-01, 0.0 /
      DATA (TSTFDN(I,4,2), I = 1, 2) / 0.0, 2.68008E-02 /
      DATA (TSTFUP(I,4,2), I = 1, 2) / 2.25915E-01, 0.0 /
      DATA (TSTDFD(I,4,2), I = 1, 2) / 2*0.0 /
      DATA ((TSTUU(I,J,1,4,2), J = 1, 6), I = 1, 2)
     &  / 3*0.0, 3.64010E-01, 8.26993E-02, 4.92370E-02,
     &    1.05950E-02, 7.69337E-03, 3.79276E-03, 3*0.0 /


c ********************* Test Case 3A *********************************

      DATA (TSTFIR(I,1,3), I = 1, 2) / 3.14159E+00, 1.15573E+00 /
      DATA (TSTFDN(I,1,3), I = 1, 2) / 0.0, 1.73849E+00 /
      DATA (TSTFUP(I,1,3), I = 1, 2) / 2.47374E-01, 0.0 /
      DATA (TSTDFD(I,1,3), I = 1, 2) / 0.0, 0.0 /
      DATA ((TSTUU(I,J,1,1,3), J = 1, 6), I = 1, 2) /
     &    3*0.0, 1.51159E-01, 1.01103E-01, 3.95460E-02,
     &    3.05855E+00, 2.66648E-01, 2.13750E-01, 3*0.0 /

c ********************* Test Case 3B *********************************

      DATA (TSTFIR(I,2,3), I = 1, 2) / 3.14159E+00, 1.05389E-03 /
      DATA (TSTFDN(I,2,3), I = 1, 2) / 0.0, 1.54958E+00 /
      DATA (TSTFUP(I,2,3), I = 1, 2) / 1.59096E+00, 0.0 /
      DATA (TSTDFD(I,2,3), I = 1, 2) / 2*0.0 /
      DATA ((TSTUU(I,J,1,2,3), J = 1, 6), I = 1, 2) /
     &    3*0.0, 3.79740E-01, 5.19598E-01, 4.93302E-01,
     &    6.69581E-01, 4.22350E-01, 2.36362E-01, 3*0.0 /


c ********************* Test Case 4A *********************************

      DATA (TSTFIR(I,1,4), I = 1, 3)
     &  / 3.14159E+00, 1.90547E+00, 1.15573E+00 /
      DATA (TSTFDN(I,1,4), I = 1, 3)
     &  / 0.0, 1.17401E+00, 1.81264E+00 /
      DATA (TSTFUP(I,1,4), I = 1, 3)
     &  / 1.73223E-01, 1.11113E-01, 0.0 /
      DATA (TSTDFD(I,1,4), I = 1, 3) / 3*0.0 /
      DATA ((TSTUU(I,J,1,1,4), J = 1, 6), I = 1, 3)
     &  / 3*0.0, 9.26837E-02,
     &    6.59569E-02, 3.64755E-02, 2.51608E+00, 1.19287E-01,
     &    1.34962E-01, 1.23887E-01, 4.02058E-02, 1.77746E-02,
     &    3.37302E+00, 2.19835E-01, 1.56893E-01, 3*0.0 /

c ********************* Test Case 4B *********************************

      DATA (TSTFIR(I,2,4), I = 1, 3)
     &  / 3.14159E+00, 1.90547E+00, 1.15573E+00 /
      DATA (TSTFDN(I,2,4), I = 1, 3)
     &  / 0.0, 1.01517E+00, 1.51554E+00 /
      DATA (TSTFUP(I,2,4), I = 1, 3)
     &  / 1.23665E-01, 7.88690E-02, 0.0 /
      DATA (TSTDFD(I,2,4), I = 1, 3)
     &  / 3.43724E-01, 3.52390E-01, 3.19450E-01 /
      DATA ((TSTUU(I,J,1,2,4), J = 1, 6), I = 1, 3)
     &  / 3*0.0, 6.53056E-02,
     &    4.55144E-02, 2.82693E-02, 2.24258E+00, 9.66049E-02,
     &    9.61335E-02, 8.43278E-02, 2.79473E-02, 1.38835E-02,
     &    2.97057E+00, 1.67698E-01, 1.08115E-01, 3*0.0 /

c ********************* Test Case 4C *********************************

      DATA (TSTFIR(I,3,4), I = 1, 3)
     &  / 1.57080E+00, 5.77864E-01, 2.12584E-01 /
      DATA (TSTFDN(I,3,4), I = 1, 3)
     &  / 0.0, 7.02764E-01, 8.03294E-01 /
      DATA (TSTFUP(I,3,4), I = 1, 3)
     &  / 2.25487E-01, 1.23848E-01, 0.0 /
      DATA (TSTDFD(I,3,4), I = 1, 3)
     &  / 3.85003E-01, 3.37317E-01, 2.16403E-01 /
      DATA (((TSTUU(I,J,K,3,4), J = 1, 6), I = 1, 3), K = 1, 3)
     &  / 3*0.0, 8.70812E-01,
     &    2.24960E-01, 2.27572E-02, 4.77016E-02, 3.02631E+00,
     &    1.41195E+00, 6.97692E-01, 1.09130E-01, 9.32861E-03,
     &    8.38488E-02, 2.70538E+00, 8.76523E-01, 6*0.0,
     &    8.88117E-02, 5.77411E-02, 2.27572E-02,
     &    4.77016E-02, 5.80971E-02, 1.04502E-01, 9.16071E-02,
     &    2.95842E-02, 9.32861E-03, 8.38488E-02, 9.42187E-02,
     &    8.95457E-02, 6*0.0, 6.98247E-02,
     &    5.02877E-02, 2.27572E-02, 4.77016E-02, 2.58544E-02,
     &    6.25954E-02, 5.91273E-02, 2.47702E-02, 9.32861E-03,
     &    8.38488E-02, 3.99383E-02, 4.67155E-02, 3*0.0 /


c ********************* Test Case 5A *********************************

      DATA (TSTFIR(I,1,5), I = 1, 3)
     &  / 3.14159E+00, 3.97856E-14, 5.03852E-28 /
      DATA (TSTFDN(I,1,5), I = 1, 3)
     &  / 0.0, 2.24768E+00, 4.79851E-01 /
      DATA (TSTFUP(I,1,5), I = 1, 3)
     &  / 2.66174E+00, 1.76783E+00, 0.0 /
      DATA (TSTDFD(I,1,5), I = 1, 3) / 3*0.0 /
      DATA ((TSTUU(I,J,1,1,5), J = 1, 6), I = 1, 3)
     &  / 3*0.0, 4.58927E-01,
     &    7.72983E-01, 1.07196E+00, 7.53662E-01, 6.96362E-01,
     &    6.50541E-01, 6.27631E-01, 5.81809E-01, 5.24532E-01,
     &    1.95230E-01, 1.31990E-01, 7.20655E-02, 3*0.0 /

c ********************* Test Case 5B *********************************

      DATA (TSTFIR(I,2,5), I = 1, 3)
     &  / 1.28058E-01, 8.67322E-06, 4.47729E-21 /
      DATA (TSTFDN(I,2,5), I = 1, 3)
     &  / 1.74767E+00, 2.33975E-01, 6.38345E-05 /
      DATA (TSTFUP(I,2,5), I = 1, 3)
     &  / 2.70485E-01, 3.74252E-02, 1.02904E-05 /
      DATA (TSTDFD(I,2,5), I = 1, 3)
     &  / 3.10129E-01, 4.52671E-02, 1.25021E-05 /
      DATA ((TSTUU(I,J,1,2,5), J = 1, 6), I = 1, 3)
     &  / 6.79623E+01, 2.21027E-01, 1.36619E-01, 1.14084E-01,
     &    8.73870E-02, 8.81626E-02, 2.05706E-01, 4.92736E-02,
     &    2.65449E-02, 2.02154E-02, 1.29661E-02, 9.51334E-03,
     &    3.41286E-05, 1.39916E-05, 7.47039E-06, 5.65602E-06,
     &    3.58245E-06, 2.57858E-06 /

c ********************* Test Case 6A *********************************

      DATA (TSTFIR(I,1,6), I = 1, 2) / 2*100.0 /
      DATA (TSTFDN(I,1,6), I = 1, 2) / 2*0.0 /
      DATA (TSTFUP(I,1,6), I = 1, 2) / 2*0.0 /
      DATA (TSTDFD(I,1,6), I = 1, 2) / 2*200.0 /
      DATA ((TSTUU(I,J,1,1,6), J = 1, 4), I = 1, 2) / 8*0.0 /

c ********************* Test Case 6B *********************************

      DATA (TSTFIR(I,2,6), I = 1, 3)
     &  / 1.000000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,2,6), I = 1, 3) / 3*0.0 /
      DATA (TSTFUP(I,2,6), I = 1, 3) / 3*0.0 /
      DATA (TSTDFD(I,2,6), I = 1, 3)
     &  / 2.00000E+02, 7.35759E+01, 2.70671E+01 /
      DATA ((TSTUU(I,J,1,2,6), J = 1, 4), I = 1, 3) / 12*0.0 /

c ********************* Test Case 6C *********************************

      DATA (TSTFIR(I,3,6), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,3,6), I = 1, 3) / 3*0.0 /
      DATA (TSTFUP(I,3,6), I = 1, 3)
     &  / 1.48450E+00, 2.99914E+00, 6.76676E+00 /
      DATA (TSTDFD(I,3,6), I = 1, 3)
     &  / 2.02010E+02, 7.79962E+01, 4.06006E+01 /
      DATA ((TSTUU(I,J,1,3,6), J = 1, 4), I = 1, 3)
     &  / 2*0.0, 9.77882E-05, 7.92386E-01,
     &    2*0.0, 1.45131E-02, 1.30642E+00,
     &    2*0.0, 2.15393E+00, 2.15393E+00 /

c ********************* Test Case 6D *********************************

      DATA (TSTFIR(I,4,6), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,4,6), I = 1, 3) / 3*0.0 /
      DATA (TSTFUP(I,4,6), I = 1, 3)
     &  / 6.70783E-01, 1.39084E+00, 3.31655E+00 /
      DATA (TSTDFD(I,4,6), I = 1, 3)
     &  / 2.00936E+02, 7.57187E+01, 3.45317E+01 /
      DATA ((TSTUU(I,J,1,4,6), J = 1, 4), I = 1, 3)
     &  / 2*0.0, 6.80068E-05, 3.15441E-01,
     &    2*0.0, 1.00931E-02, 5.20074E-01,
     &    2*0.0, 1.49795E+00, 8.57458E-01 /

c ********************* Test Case 6E *********************************

      DATA (TSTFIR(I,5,6), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,5,6), I = 1, 3) / 3*0.0 /
      DATA (TSTFUP(I,5,6), I = 1, 3)
!     &  / 7.95458E+01, 1.59902E+02, 3.56410E+02 /
     &  / 9.44651E+01, 1.90616E+02, 4.28806E+02 /
      DATA (TSTDFD(I,5,6), I = 1, 3)
!     &  / 3.07079E+02, 3.07108E+02, 7.17467E+02 /
     &  / 3.27710E+02, 3.53897E+02, 8.78110E+02 /
      DATA ((TSTUU(I,J,1,5,6), J = 1, 4), I = 1, 3)
!     &  / 2*0.0, 4.53789E-03, 4.33773E+01,
!     &    2*0.0, 6.73483E-01, 7.15170E+01,
!     &    2*0.0, 9.99537E+01, 1.17912E+02 /
     &  / 2*0.0, 6.01546E-03, 5.06858E+01,
     &    2*0.0, 8.92774E-01, 8.35668E+01,
     &    2*0.0, 1.32499E+02, 1.37778E+02 /

c ********************* Test Case 6F *********************************

      DATA (TSTFIR(I,6,6), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,6,6), I = 1, 3)
     &  / 3.21497E+02, 1.42493E+02, 7.05305E+01 /
      DATA (TSTFUP(I,6,6), I = 1, 3)
!    &  / 8.27917E+01, 1.66532E+02, 3.71743E+02 /
     &  / 9.77110E+01, 1.97247E+02, 4.41139E+02 /
      DATA (TSTDFD(I,6,6), I = 1, 3)
!    &  / 9.54523E+02, 5.27085E+02, 8.45341E+02 /
     &  / 9.75154E+02, 5.73874E+02, 1.00598E+03 /
      DATA ((TSTUU(I,J,1,6,6), J = 1, 4), I = 1, 3)
!    &  / 1.02336E+02, 1.02336E+02, 4.80531E-03, 4.50168E+01,
!    &    6.20697E+01, 6.89532E-01, 7.13172E-01, 7.42191E+01,
!    &    3.76472E+01, 4.64603E-03, 1.05844E+02, 1.22368E+02 /
     &  / 1.02336E+02, 1.02336E+02, 6.28289E-03, 5.23253E+01,
     &    6.20697E+01, 6.89532E-01, 9.32463E-01, 8.62699E+01,
     &    3.76472E+01, 4.64603E-03, 1.38390E+02, 1.42235E+02 /

c ********************* Test Case 6G *********************************

      DATA (TSTFIR(I,7,6), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,7,6), I = 1, 3)
     &  / 3.21497E+02, 3.04775E+02, 3.63632E+02 /
      DATA (TSTFUP(I,7,6), I = 1, 3)
!    &  / 3.35292E+02, 4.12540E+02, 4.41125E+02 /
     &  / 3.50211E+02, 4.43254E+02, 5.13521E+02 /
      DATA (TSTDFD(I,7,6), I = 1, 3)
!    &  / 5.80394E+02, 1.27117E+02, -1.68003E+02 /
     &  / 6.01025E+02, 1.73906E+02, -7.36023E+00 /
      DATA ((TSTUU(I,J,1,7,6), J = 1, 4), I = 1, 3)
!    &  / 1.02336E+02, 1.02336E+02, 7.80733E+01, 1.16430E+02,
!    &    9.78748E+01, 1.01048E+02, 1.15819E+02, 1.34966E+02,
!    &    1.10061E+02, 1.38631E+02, 1.38695E+02, 1.40974E+02 /
     &  / 1.02336E+02, 1.02336E+02, 7.80748E+01, 1.23739E+02,
     &    9.78748E+01, 1.01048E+02, 1.16039E+02, 1.47015E+02,
     &    1.10061E+02, 1.38631E+02, 1.71240E+02, 1.60840E+02 /

c ********************* Test Case 6H *********************************

      DATA (TSTFIR(I,8,6), I = 1, 3)
     &  / 1.00000E+02, 1.35335E+01, 2.06115E-07 /
      DATA ( TSTFDN(I,8,6), I = 1, 3)
     &  / 3.21497E+02, 2.55455E+02, 4.43444E+02 /
      DATA (TSTFUP(I,8,6), I = 1, 3)
!     &  / 2.37350E+02, 2.61130E+02, 4.55861E+02 /
     &  / 2.37351E+02, 2.61131E+02, 5.28258E+02 /
      DATA (TSTDFD(I,8,6), I = 1, 3)
!     &  / 4.23780E+02, 6.19828E+01, -3.11658E+01 /
     &  / 4.23780E+02, 6.19842E+01, 1.29477E+02  /
      DATA ((TSTUU(I,J,1,8,6), J = 1, 4), I = 1, 3)
!     &  / 1.02336E+02, 1.02336E+02, 7.12616E+01, 7.80736E+01,
!     &    8.49992E+01, 7.73186E+01, 7.88310E+01, 8.56423E+01,
!     &    1.38631E+02, 1.45441E+02, 1.44792E+02, 1.45163E+02 /
     &  / 1.02336E+02, 1.02336E+02, 7.12616E+01, 7.80745E+01,
     &    8.49992E+01, 7.73186E+01, 7.88310E+01, 8.56448E+01,
     &    1.38631E+02, 1.45441E+02, 1.77337E+02, 1.65030E+02 /


c ********************* Test Case 7A *********************************

      DATA (TSTFIR(I,1,7), I = 1, 2) / 2*0.0 /
      DATA (TSTFDN(I,1,7), I = 1, 2) / 0.0, 1.21204E+02 /
      DATA (TSTFUP(I,1,7), I = 1, 2) / 8.62936E+01,  0.0 /
      DATA (TSTDFD(I,1,7), I = 1, 2) /-5.13731E+01,-5.41036E+02 /

c ********************* Test Case 7B *********************************

      DATA (TSTFIR(I,2,7), I = 1, 2) / 2*0.0 /
      DATA (TSTFDN(I,2,7), I = 1, 2) / 0.0, 2.07786E-05 /
      DATA (TSTFUP(I,2,7), I = 1, 2) / 1.10949E-06,  0.0 /
      DATA (TSTDFD(I,2,7), I = 1, 2) / 8.23219E-08, -5.06461E-06 /
      DATA ((TSTUU(I,J,1,2,7), J = 1, 2), I = 1, 2) /
     &    0.00000E+00, 4.65744E-07, 7.52311E-06, 0.00000E+00 /

c ********************* Test Case 7C *********************************

      DATA (TSTFIR(I,3,7), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,3,7), I = 1, 3)
     &  / 3.19830E+02, 3.54099E+02, 3.01334E+02 /
      DATA (TSTFUP(I,3,7), I = 1, 3)
     &  / 4.29572E+02, 4.47018E+02, 5.94576E+02 /
      DATA (TSTDFD(I,3,7), I = 1, 3)
     &  /-8.04270E+01, 2.51589E+02, 7.15964E+02 /
      DATA (((TSTUU(I,J,K,3,7), J = 1, 4), I = 1, 3), K = 1, 2)
     &  / 1.01805E+02, 1.01805E+02, 1.46775E+02, 1.49033E+02,
     &    1.06583E+02, 1.28565E+02, 1.04464E+02, 1.59054E+02,
     &    9.66519E+01, 8.65854E+01, 1.89259E+02, 1.89259E+02,
     &    1.01805E+02, 1.01805E+02, 1.29641E+02, 1.49033E+02,
     &    1.06583E+02, 1.06408E+02, 9.48418E+01, 1.59054E+02,
     &    9.66519E+01, 7.49310E+01, 1.89259E+02, 1.89259E+02 /

c ********************* Test Case 7D *********************************

      DATA (TSTFIR(I,4,7), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,4,7), I = 1, 3)
     &  / 3.19830E+02, 3.50555E+02, 2.92063E+02 /
      DATA (TSTFUP(I,4,7), I = 1, 3)
     &  / 3.12563E+02, 2.68126E+02, 3.05596E+02 /
      DATA (TSTDFD(I,4,7), I = 1, 3)
     &  /-1.68356E+02, 1.01251E+02, 4.09326E+02 /
      DATA (((TSTUU(I,J,K,4,7), J = 1, 4), I = 1, 3), K = 1, 2)
     &  / 1.01805E+02, 1.01805E+02, 1.40977E+02, 9.62764E+01,
     &    1.06203E+02, 1.23126E+02, 9.19545E+01, 8.89528E+01,
     &    9.56010E+01, 7.25576E+01, 9.72743E+01, 9.72743E+01,
     &    1.01805E+02, 1.01805E+02, 1.23843E+02, 9.62764E+01,
     &    1.06203E+02, 1.00969E+02, 8.23318E+01, 8.89528E+01,
     &    9.56010E+01, 6.09031E+01, 9.72743E+01, 9.72743E+01 /

c ********************* Test Case 7E *********************************

      DATA (TSTFIR(I,5,7), I = 1, 3)
     &  / 1.00000E+02, 3.67879E+01, 1.35335E+01 /
      DATA (TSTFDN(I,5,7), I = 1, 3)
!    &  / 3.19830E+02, 3.53275E+02, 2.99002E+02 /
     &  / 3.19830E+02, 3.54468E+02, 3.02366E+02 /
      DATA (TSTFUP(I,5,7), I = 1, 3)
!    &  / 4.04300E+02, 4.07843E+02, 5.29248E+02 /
     &  / 4.40940E+02, 4.64624E+02, 6.23842E+02 /
      DATA (TSTDFD(I,5,7), I = 1, 3)
!    &  /-9.98568E+01, 2.17387E+02, 6.38461E+02 /
     &  /-7.16977E+01, 2.66918E+02, 7.50170E+02 /
      DATA (((TSTUU(I,J,K,5,7), J = 1, 4), I = 1, 3), K = 1, 2)
!    &  / 1.01805E+02, 1.01805E+02, 1.45448E+02, 1.38554E+02,
!    &    1.06496E+02, 1.27296E+02, 1.01395E+02, 1.45229E+02,
!    &    9.63993E+01, 8.29009E+01, 1.60734E+02, 1.71307E+02,
!    &    1.01805E+02, 1.01805E+02, 1.28281E+02, 1.38554E+02,
!    &    1.06496E+02, 1.05111E+02, 9.16726E+01, 1.45229E+02,
!    &    9.63993E+01, 7.11248E+01, 1.59286E+02, 1.71307E+02 /
     &  / 1.01805E+02, 1.01805E+02, 1.47357E+02, 1.53713E+02,
     &    1.06621E+02, 1.29120E+02, 1.05792E+02, 1.65216E+02,
     &    9.67644E+01, 8.81547E+01, 2.00886E+02, 1.97240E+02,
     &    1.01805E+02, 1.01805E+02, 1.30246E+02, 1.53713E+02,
     &    1.06621E+02, 1.06984E+02, 9.62371E+01, 1.65216E+02,
     &    9.67644E+01, 7.65889E+01, 2.01922E+02, 1.97240E+02 /

c ********************* Test Case 8A *********************************

      DATA (TSTFIR(I,1,8), I = 1, 3) / 3*0.0 /
      DATA (TSTFDN(I,1,8), I = 1, 3) /
     &  1.00000E+00, 7.22235E-01, 5.13132E-01 /
      DATA (TSTFUP(I,1,8), I = 1, 3) / 9.29633E-02, 2.78952E-02, 0.0 /
      DATA (TSTDFD(I,1,8), I = 1, 3) /
     &  1.12474E+00, 6.51821E-01, 5.63361E-01 /
      DATA ((TSTUU(I,J,1,1,8), J = 1, 4), I = 1, 3) /
     &  2*3.18310E-01, 5.62566E-02, 1.94423E-02,
     &  2.62711E-01, 1.36952E-01, 1.84909E-02, 5.52188E-03,
     &  2.10014E-01, 5.60376E-02, 2*0.0 /

c ********************* Test Case 8B *********************************

      DATA (TSTFIR(I,2,8), I = 1, 3) / 3*0.0 /
      DATA (TSTFDN(I,2,8), I = 1, 3) /
     &  1.00000E+00, 7.95332E-01, 6.50417E-01 /
      DATA (TSTFUP(I,2,8), I = 1, 3) / 2.25136E-01, 1.26349E-01, 0.0 /
      DATA (TSTDFD(I,2,8), I = 1, 3) /
     &  5.12692E-01, 3.56655E-01, 5.68095E-02 /
      DATA ((TSTUU(I,J,1,2,8), J = 1, 4), I = 1, 3) /
     &  2*3.18310E-01, 1.23687E-01, 4.95581E-02,
     &  2.77499E-01, 1.83950E-01, 8.35695E-02, 2.50575E-02,
     &  2.40731E-01, 1.29291E-01, 2*0.0 /

c ********************* Test Case 8C *********************************

      DATA (TSTFIR(I,3,8), I = 1, 3) / 3*0.0 /
      DATA (TSTFDN(I,3,8), I = 1, 3) /
     &  1.00000E+00, 4.86157E-01, 1.59984E-01 /
      DATA (TSTFUP(I,3,8), I = 1, 3) / 3.78578E-01, 2.43397E-01, 0.0 /
      DATA (TSTDFD(I,3,8), I = 1, 3) /
     &  5.65095E-01, 2.76697E-01, 1.35679E-02 /
      DATA ((TSTUU(I,J,1,3,8), J = 1, 4), I = 1, 3) /
     &  2*3.18310E-01, 1.49335E-01, 1.04766E-01,
     &  1.89020E-01, 9.88158E-02, 9.65192E-02, 6.54445E-02,
     &  6.84762E-02, 2.96698E-02, 2*0.0 /


c ********************* Test Case 9A *********************************

      DATA (TSTFIR(I,1,9), I = 1, 5) / 5*0.0 /
      DATA (TSTFDN(I,1,9), I = 1, 5) /
     &  1.00000E+00, 3.55151E-01, 1.44265E-01, 6.71445E-03, 6.16968E-07/
      DATA (TSTFUP(I,1,9), I = 1, 5) /
     &  2.27973E-01, 8.75098E-02, 3.61819E-02, 2.19291E-03, 0.0 /
      DATA (TSTDFD(I,1,9), I = 1, 5) /
     &  8.82116E-01, 2.32366E-01, 9.33443E-02, 3.92782E-03, 1.02500E-07/
      DATA ((TSTUU(I,J,1,1,9), J = 1, 4), I = 1, 5) /
     &  2*3.18310E-01, 9.98915E-02, 5.91345E-02,
     &  1.53507E-01, 5.09531E-02, 3.67006E-02, 2.31903E-02,
     &  7.06614E-02, 2.09119E-02, 1.48545E-02, 9.72307E-03,
     &  3.72784E-03, 1.08815E-03, 8.83316E-04, 5.94743E-04,
     &  2.87656E-07, 1.05921E-07, 2*0.0 /

c ********************* Test Case 9B *********************************

      DATA (TSTFIR(I,2,9), I = 1, 5) / 5*0.0 /
      DATA (TSTFDN(I,2,9), I = 1, 5) /
     &  1.00000E+00, 4.52357E-01, 2.36473E-01, 2.76475E-02, 7.41853E-05/
      DATA (TSTFUP(I,2,9), I = 1, 5) /
     &  1.00079E-01, 4.52014E-02, 2.41941E-02, 4.16016E-03, 0.0 /
      DATA (TSTDFD(I,2,9), I = 1, 5) /
     &  8.04577E-01, 2.55330E-01, 1.30976E-01, 1.36227E-02, 1.22022E-05/
      DATA ((TSTUU(I,J,1,2,9), J = 1, 4), I = 1, 5) /
     &  2*3.18310E-01, 7.39198E-02, 1.32768E-02,
     &  1.96609E-01, 5.92369E-02, 3.00230E-02, 7.05566E-03,
     &  1.15478E-01, 3.01809E-02, 1.52672E-02, 4.06932E-03,
     &  1.46177E-02, 3.85590E-03, 2.38301E-03, 7.77890E-04,
     &  3.37742E-05, 1.20858E-05, 2*0.0 /

c ********************* Test Case 9C *********************************

      DATA (TSTFIR(I,3,9), I = 1, 5) /
     &  1.57080E+00, 1.92354E-01, 2.35550E-02, 9.65131E-06, 9.03133E-19/
      DATA (TSTFDN(I,3,9), I = 1, 5 ) /
     &  6.09217E+00, 4.97279E+00, 4.46616E+00, 4.22731E+00, 4.73767E+00/
      DATA (TSTFUP(I,3,9), I = 1, 5) /
     &  4.68414E+00, 4.24381E+00, 4.16941E+00, 4.30667E+00, 5.11524E+00/
      DATA (TSTDFD(I,3,9), I = 1, 5) /
     &  3.49563E+00, 8.81206E-01, 3.50053E-01, 1.93471E-02, 7.15349E-02/
      DATA (((TSTUU(I,J,K,3,9), J = 1, 4), I = 1, 5), K = 1, 3)
     & / 1.93920E+00, 1.93920E+00, 1.61855E+00, 1.43872E+00,
     &   1.66764E+00, 1.44453E+00, 1.38339E+00, 1.33890E+00,
     &   1.48511E+00, 1.35009E+00, 1.33079E+00, 1.32794E+00,
     &   1.34514E+00, 1.35131E+00, 1.35980E+00, 1.37918E+00,
     &   1.48927E+00, 1.54270E+00, 1.62823E+00, 1.62823E+00,
     &   1.93920E+00, 1.93920E+00, 1.57895E+00, 1.43872E+00,
     &   1.66764E+00, 1.42925E+00, 1.37317E+00, 1.33890E+00,
     &   1.48511E+00, 1.34587E+00, 1.32921E+00, 1.32794E+00,
     &   1.34514E+00, 1.35129E+00, 1.35979E+00, 1.37918E+00,
     &   1.48927E+00, 1.54270E+00, 1.62823E+00, 1.62823E+00,
     &   1.93920E+00, 1.93920E+00, 1.56559E+00, 1.43872E+00,
     &   1.66764E+00, 1.42444E+00, 1.37034E+00, 1.33890E+00,
     &   1.48511E+00, 1.34469E+00, 1.32873E+00, 1.32794E+00,
     &   1.34514E+00, 1.35128E+00, 1.35979E+00, 1.37918E+00,
     &   1.48927E+00, 1.54270E+00, 2*1.62823E+00 /

c *********************Test Case 15a *********************************

      DATA (TSTFIR(I,1,15), I = 1, 3) /
     & 8.66025E-01, 5.45681E-01, 4.13603E-01 / 
      DATA (TSTFDN(I,1,15), I = 1, 3 ) /
     & 0.00000E-00, 2.21186E-01, 3.38138E-01 /
      DATA (TSTFUP(I,1,15), I = 1, 3 ) /
     & 2.77760E-01, 1.78593E-01, 1.63465E-01 /
      DATA (TSTDFD(I,1,15), I = 1, 3 ) /
     & 1.91452E-06, 1.80279E-06, 1.74169E-06 /
      DATA (((TSTUU(I,J,K,1,15), J = 1, 4), I = 1, 3), K = 1, 3)
     & / 1.23554E-01, 1.12727E-01, 8.33376E-02, 7.88616E-02,
     &   9.78420E-02, 7.64028E-02, 5.40164E-02, 5.13113E-02,
     &   5.55814E-02, 5.30299E-02, 4.79406E-02, 4.69991E-02,
     &   1.13583E-01, 1.07286E-01, 8.90993E-02, 7.88616E-02,
     &   7.58666E-02, 6.87194E-02, 5.70572E-02, 5.13113E-02,
     &   6.41686E-02, 6.06886E-02, 5.34627E-02, 4.69992E-02,
     &   1.31842E-01, 1.26113E-01, 1.10240E-01, 7.88616E-02,
     &   7.67841E-02, 7.36885E-02, 6.89669E-02, 5.13113E-02,
     &   7.39453E-02, 6.96550E-02, 6.10954E-02, 4.69991E-02 /





c *********************Test Case 15b *********************************
      DATA (TSTFIR(I,2,15), I = 1, 3) /
     & 8.66025E-01, 5.45681E-01, 4.13603E-01 / 
      DATA (TSTFDN(I,2,15), I = 1, 3 ) /
     & 0.00000E-00, 1.90697E-01, 3.02160E-01 /
      DATA (TSTFUP(I,2,15), I = 1, 3 ) /
     & 1.73025E-01, 4.33759E-02, 2.27533E-02 /
      DATA (TSTDFD(I,2,15), I = 1, 3 ) /
     & 1.68943E-06, 1.38905E-06, 1.27981E-06 /
      DATA (((TSTUU(I,J,K,2,15), J = 1, 4), I = 1, 3), K = 1, 3)
     & / 1.05039E-01, 9.19789E-02, 5.96677E-02, 4.42653E-02,
     &   7.28008E-02, 5.02279E-02, 2.72793E-02, 1.14839E-02,
     &   3.48211E-02, 2.79014E-02, 2.21136E-02, 7.09119E-03,
     &   9.44399E-02, 8.32705E-02, 5.47476E-02, 4.42653E-02,
     &   3.86983E-02, 2.55378E-02, 9.51889E-03, 1.14839E-02,
     &   1.37107E-02, 9.44994E-03, 3.46688E-03, 7.09119E-03,
     &   1.12580E-01, 1.00789E-01, 7.18390E-02, 4.42653E-02,
     &   3.43420E-02, 2.33428E-02, 1.37850E-02, 1.14839E-02,
     &   1.17920E-02, 8.06508E-03, 2.79583E-03, 7.09119E-03 /



c *********************Test Case 15c *********************************
      DATA (TSTFIR(I,3,15), I = 1, 3) /
     & 8.66025E-01, 5.45681E-01, 4.13603E-01 / 
      DATA (TSTFDN(I,3,15), I = 1, 3 ) /
     & 0.00000E-00, 2.14201E-01, 3.30068E-01 /
      DATA (TSTFUP(I,3,15), I = 1, 3 ) /
     & 2.54555E-01, 1.48380E-01, 1.32172E-01 /
      DATA (TSTDFD(I,3,15), I = 1, 3 ) /
     & 1.86327E-06, 1.70778E-06, 1.65841E-06 /
      DATA (((TSTUU(I,J,K,3,15), J = 1, 4), I = 1, 3), K = 1, 3)
     & / 1.19166E-01, 1.07082E-01, 7.41418E-02, 7.26226E-02,
     &   9.00393E-02, 6.59806E-02, 4.05345E-02, 4.45512E-02,
     &   4.94497E-02, 4.05951E-02, 3.33681E-02, 4.04333E-02,
     &   1.09205E-01, 1.01727E-01, 8.06139E-02, 7.26226E-02,
     &   6.84651E-02, 5.86094E-02, 4.48932E-02, 4.45511E-02,
     &   5.82113E-02, 4.82087E-02, 4.03103E-02, 4.04332E-02,
     &   1.27564E-01, 1.20988E-01, 1.03628E-01, 7.26226E-02,
     &   7.10388E-02, 6.56529E-02, 6.04316E-02, 4.45511E-02,
     &   7.13241E-02, 5.98879E-02, 5.21536E-02, 4.04332E-02 /

c *********************Test Case 15d *********************************
      DATA (TSTFIR(I,4,15), I = 1, 3) /
     & 8.66025E-01, 5.45681E-01, 4.13603E-01 / 
      DATA (TSTFDN(I,4,15), I = 1, 3 ) /
     & 0.00000E-00, 2.25389E-01, 3.42092E-01 /
      DATA (TSTFUP(I,4,15), I = 1, 3 ) /
     & 2.97775E-01, 2.02805E-01, 1.87433E-01 /
      DATA (TSTDFD(I,4,15), I = 1, 3 ) /
     & 1.95071E-06, 1.84998E-06, 1.77781E-06 /
      DATA (((TSTUU(I,J,K,4,15), J = 1, 4), I = 1, 3), K = 1, 3)
     & / 1.26062E-01, 1.14608E-01, 8.79932E-02, 8.99278E-02,
     &   9.34865E-02, 7.33495E-02, 6.02902E-02, 6.54998E-02,
     &   2.78309E-02, 4.54440E-02, 5.49975E-02, 6.19027E-02,
     &   1.16114E-01, 1.09216E-01, 9.29265E-02, 8.99278E-02,
     &   7.30116E-02, 6.61578E-02, 6.17029E-02, 6.54998E-02,
     &   4.54126E-02, 5.41580E-02, 5.85154E-02, 6.19027E-02,
     &   1.34402E-01, 1.28103E-01, 1.13285E-01, 8.99278E-02,
     &   7.57499E-02, 7.17037E-02, 7.20841E-02, 6.54998E-02,
     &   6.43650E-02, 6.40986E-02, 6.42596E-02, 6.19027E-02 /




      END
